/*
============SAS CODE FOR RUG-IV CODE VERSION 1.03.0======================== 

=========================DOCUMENTATION=====================================
---------------------------------------------------------------------------
 General documentation about the RUG-IV Version 1.03.0 SAS Code is provided
   in comments below and in the document 
   
        "RUGIV SAS user doc V1.03.0 yyyymmdd.pdf" 
   
   in the RUG IV grouper package (where "yyyymmdd" stands for the 
   publication date).

 Description of the RUG-IV Classification System can be found in Chapter 6
   of the RAI Manual for the MDS 3.0. 

--------------------------------------------------------------------------- 
CHANGES WITH SAS CODE VERSION 1.03.0:
These are the changes since the last public SAS code version 1.02.1:
   1. Add a new value to rehabilitation classification type parameter
      (sRehabType).  This new value of MCAR3 for FY2014 Medicare 
      classification works the same as the MCAR2 (FY2012 and FY2013 Medicare 
      classification) value but changes the classification requirements 
      for Medium Rehabilitation and Low Rehabilitation as follows:
      a.  For Medium Rehabilitation, the requirement of 5 or more therapy 
          days across the three discipline (sum of O0410A4, O0400B4, and 
          O0400C4) is being replaced by 5 or more distinct calendar days 
          of therapy (O0420).
      b.  For Low Rehabilitation, the requirement of 3 or more therapy 
          days across the three discipline (sum of O0410A4, O0400B4, and 
          O0400C4) is being replaced by 3 or more distinct calendar days 
          of therapy (O0420).
   2. As of October 1, 2013, the MDS 3.0 nutrition item K0700A indicating the
      proportion of calories by artificial route in the last 7 days is being 
      replaced by three new calorie proportion items K0710A1 (received while 
      not a resident), K0710A2 (received while a resident), and K0710A3 
      (received in the entire 7 days).  For assessments with assessment 
      reference date (A2300) on or after 10/1/2013, K0700A is being replaced 
      by K0710A3 in the RUG classification logic.
   3. As of October 1, 2013, he MDS 3.0 nutrition item K0700B indicating the
      average fluid intake by artificial route in the last 7 days is being 
      replaced by three new fluid intake items K0710B1 (received while 
      not a resident), K0710B2 (received while a resident), and K0710B3 
      (received in the entire 7 days).  For assessments with assessment 
      reference date (A2300) on or after 10/1/2013, K0700B is being replaced 
      by K0710B3 in the RUG classification logic.

      Version 1.03.0 is backwardly compatible with the prior version 1.02.1
      and it is not necessary to use version 1.02.1 for assessments with
      A2300 before 10/1/2013 and a separate version 1.03.0 for assessments 
      with A2300 on or after 10/1/2013. Version 1.03.0 will handle 
      assessments both before 10/1/2013 and on or after
      10/1/2013. 
 
NOTE:  Code segments changed with version 1.03.0 are indicated by comments.

--------------------------------------------------------------------------- 
SYNTAX USED IN THIS SAS CODE:
   1.  All text between a slash (/) asterisk (*) combination and the next 
       asterisk (*) slash (/) combination is a comment. 
   2.  IF an asterisk (*) is the first nonblank character in a line then this
         is a comment that continues until the next semicolon (;).  
   3   All statements start with a character other than an asterisk (*) and end
         with the next semicolon (;).
   4.  All command words are represented in capital letters.
   5.  All MDS 3.0 variables use the standard labels from the MDS 3.0 Data 
         Specifications (e.g. G0110A1 for Bed Mobility Self-Performance).
         ALL MDS 3.0 VARIABLES ARE REQUIRED TO BE CHARACTER VARIABLES IN THIS 
         SAS CODE.  MDS 3.0 date items are formatted YYYYMMDD.
   6.  The RUG-IV SAS code Version 1.03.0 input parameters use the standard 
         names found in RUG IV SAS user documentation referenced above 
         (also listed below).
   7.  All local (non-MDS) variables use the following prefix convention:
         i_  for variables with integer values
         n_  for variables with numeric values that may contain decimals
         n   for numeric RUG group codes
         l_  for logical (0 = no, 1 = yes) variables
         s   for string (character)parameters, and character RUG group
                  codes 
         s_  for string (character) variables
         d_  for SAS date variables
   8.  The only command words and structures used in this SAS code are:
         a. IF <logical condition1>
              THEN <arithmetic statement> executed if condition1 true
            ELSE IF <logical condition2>
              THEN <arithmetic statement> executed if condition2 true
                DO <multiple statements>  END
            ELSE   <arithmetic statement> executed if no condition is true
         b. Logical operators used in logical conditions are
                =       (equal)
                >       (greater than)
                <       (less than)
                >=      (greater than or equal to)
                <=      (less than or equal to)
                NOT =   (not equal to)
         c. Relational operators used in logical conditions are
                AND    (logical and)
                OR     (logical or)
         d. INPUT(s_var,#.) -- the input function used with a numeric format
              represented by "#. " (e.g., 2. or 2.0) returns the numeric value
              for a character variable (s_var).  A ?? before "#. " suppresses
              warning messages if the s_var values is not numeric.
         e. SUBSTR(s_var,n1,n2) -- returns a substring of the character 
             variable s_var, starting with the n1 character and with a
             length of n2.   
         f. MDY(n1,n2,n3) -- creates a SAS date variable from n1 (a numeric 
             variable for the month), n2 (a numeric variable for the day),
             and n3 (a numeric variable for the 4 digit year).
         g. ROUND(n1,4) -- rounds a numeric value (n1) to the fourth decimal
             place.
         h. The SAS text concatenation operator (||) is used to concatenate
             text strings.  Example:
                'a' || 'b' = 'ab'
         i. GO TO <labeled statement> -- execution jumps to the <labeled
             statement>. A labeled statement is a name followed by a
             colon (:).  The only labeled statement in this code is
             SKIPEND:.
         j. UPCASE(s_var) = the s_var string variable converted to uppercase.
   9.  Nested IF structures are indented for clarity.
  10.  Single quotes (') delimit a character constant.

--------------------------------------------------------------------------- 
PRE-PROCESSING REQUIRED BEFORE EXECUTION OF RUGIV V1.03.0 SAS:
   1.  Extract the 121 RUG-IV items from an MDS 3.0 record and create a SAS
         data set with these 121 in character format (YYYMMDD) for date items).
         The list of the 121 items, the length of each, and the start and end
         positions in the standard MDS 3.0 fixed format record.          
            "RUGIV grouper overview V1.03.0 yyyymmdd.pdf"
               and                
            "RUGIV MDS 3.0 item list V1.03.csv"
         (where "yyyymmdd" stands for the publication date).         
   2.  Set the sRehabType, sModel, and nCmiArray input parameters (described 
         below).  The nCmiArray must be defined as a 72 element array as 
         follows:
              ARRAY nCmiArray(72);
         The 72 values in nCmiArray must correspond to the 72 possible groups
         for RUG-IV (66-, 57-, and 48-group models).  The order of the groups
         in the array is that given in Table 5-1 of "RUG_IV grouper overview 
         v1.03.0 yyyymmdd.pdf" and "RUGIV group list V1.03.csv".
   3.  Read a record from the SAS data set created in step #1 and process
         with RUGIV V1.03.0.SAS.

---------------------------------------------------------------------------
OUT-OF-RANGE VALUES AND INCONSISTENT VALUES FOR MDS 3.0 ITEMS:
   If MDS 3.0 RUG-IV variables are out of range, then the SAS code may produce
     unexpected results.  Insure all MDS 3.0 RUG-IV variables are in range
     Valid ranges for these variables are available in the MDS 3.0 Data 
     Submission Specifications available on the "MDS 3.0 Technical Information"
     page on the CMS website.     
   If the MDS 3.0 Activity of Daily Living items have inconsistent values, then 
     the SAS code may produce unexpected results.  Consistency requirements
     are as follows:
       a. If either G0110A1 or G0100A2 = '8', then both must = '8'. 
       b. If either G0110B1 or G0100B2 = '8', then both must = '8'. 
       c. If either G0110H1 or G0100H2 = '8', then both must = '8'. 
       d. If either G0110I1 or G0100I2 = '8', then both must = '8'. 
   If other MDS 3.0 item values are inconsistent, then the SAS code may produce
     unexpected results.  Examples are the comatose (B0100) skip pattern, the
     rehabilitation therapy item skip patterns (items O0400A, O0400B, and 
     O0400C).
   If there are invalid date values for date items (e.g., A2300, A2400B,
     A2400C, etc.), then the SAS code may produce unexpected results.
   Inconsistent values for the MDS 3.0 RUG-IV variables are available in the 
     MDS 3.0 Data Submission Specifications available on the "MDS 3.0 Technical 
     Information" page on the CMS website.     

---------------------------------------------------------------------------
INPUT PARAMETERS USED IN THE CLASSIFICATION LOGIC:
Three input parameters must be set as follows;
   1. sRehabType (not case sensitive)
        IF sRehabType = 'MCARE', then special Medicare logic for FY2011 will 
           be used for rehabilitation classification.  This FY2011 logic 
           includes 2 features:
           a. A limit on group therapy minutes within each discipline (speech,
              occupational, and physical therapy), such that group minutes 
              cannot exceed more than 25% of total allocated minutes.  
           b. Use of the Special Medicare Short Stay rehabilitation 
              classification based on average daily minutes of therapy per 
              day rather than total minutes of therapy across days.
        IF sRehabType = 'MCAR2', then special Medicare logic for FY2012 and 
           FY2013 will be used for rehabilitation classification.  This FY2012 
           logic includes 3 features:
           a. A limit on group therapy minutes within each discipline (speech,
              occupational, and physical therapy), such that allocated group 
              minutes cannot exceed more than 25% of total allocated minutes.  
           b. Use of the Special Medicare Short Stay rehabilitation 
              classification based on average daily minutes of therapy per 
              day rather than total minutes of therapy across days.
           c. Allocation of group therapy minutes for determining total 
              rehabilitation therapy minutes.  Only 1/4 of the group therapy
              minutes are included in the total rehabilitation therapy 
              minutes.            
        IF sRehabType = 'MCAR3', then special Medicare logic for FY2014 will 
           be used for rehabilitation classification.  This FY2014 logic 
           includes 5 features:
           a. A limit on group therapy minutes within each discipline (speech,
              occupational, and physical therapy), such that allocated group 
              minutes cannot exceed more than 25% of total allocated minutes.  
           b. Use of the Special Medicare Short Stay rehabilitation 
              classification based on average daily minutes of therapy per 
              day rather than total minutes of therapy across days.
           c. Allocation of group therapy minutes for determining total 
              rehabilitation therapy minutes.  Only 1/4 of the group therapy
              minutes are included in the total rehabilitation therapy 
              minutes.
           d. For Medium Rehabilitation, the requirement of 5 or more therapy 
              days across the three discipline (sum of O0410A4, O0400B4, and 
              O0400C4) is being replaced by 5 or more distinct calendar days 
              of therapy (O0420).
           e. For Low Rehabilitation, the requirement of 3 or more therapy 
              days across the three discipline (sum of O0410A4, O0400B4, and 
              O0400C4) is being replaced by 3 or more distinct calendar days 
              of therapy (O0420).
        IF sRehabType = 'OTHER' (or any other value), then special Medicare
           logic will not be used for rehabilitation classification.
   2. sModel
        IF sModel = '66', then the standard RUG-IV 66-group model used for
           Medicare will be in effect.  The hierarchy for this model includes
           10 Rehabilitation Plus Extensive Services groups at the top level,
           followed by 14 Rehabilitation groups at the second level, and then
           the remaining groups (Extensive Services, Special Care High, etc.).
        IF sModel = '57', then the RUG-IV 57-group model (which may be used for
           Medicaid in some states) will be in effect.  In the 57-group model,
           Rehabilitation Plus Extensive Services groups are collapsed into the
           Rehabilitation groups.  The hierarchy for this model includes the
           14 Rehabilitation groups at the top level, followed by the Extensive
           Services groups, and then the remaining groups (Special Care High, 
           Special Care Low, etc.).
        IF sModel = '48', then the RUG-IV 48-group model (which may be used for
           Medicaid in some states) will be in effect.  In the 48-group model,
           the five different levels of rehabilitation therapy (ultra high,
           very high, high, medium, and low) are collapsed together.  There are 
           only 5 rehabilitation groups all involving any level of 
           rehabilitation therapy (the 5 groups being divided on the basis of 
           ADL score).  Also the Extensive Services groups are moved to the 
           top of the hierarchy. The hierarchy for this model includes the
           Extensive Services groups at the top level, followed by the 5
           collapsed Rehabilitation groups, and then the remaining groups
           (Special Care High, Special Care Low, etc.).
        IF sModel = any other value, then this SAS code will produce invalid 
           RUG-IV classification.
   3. nCmiArray
        The 72 element array nCmiArray must be defined:
             ARRAY nCmiArray (72);
        Each element of the array must be assigned a numeric value.  For example:
             nCmiArray(1) = 66.0;
        This array allows index maximizing classification.  If interest is 
        limited to hierarchical classification, then all array elements can be
        set to 0.

        The Classification Case Mix Indices (CMIs) for the 72 possible groups 
        must be in the array according to the following list of array positions
        and group codes:

          Rehabilitation/Extensive Groups for 66-Group Model	
            1 RUX
            2 RUL
            3 RVX
            4 RVL
            5 RHX
            6 RHL
            7 RMX
            8 RML
            9 RLX
          Rehabilitation Groups for 66- and 57-Group Models 
            10 RUC
            11 RUB
            12 RUA
            13 RVC
            14 RVB
            15 RVA
            16 RHC
            17 RHB
            18 RHA
            19 RMC
            20 RMB
            21 RMA
            22 RLB
            23 RLA
          Extensive Groups for All Models (66-, 57-, and 48-Groups) 
            24 ES3
            25 ES2
            26 ES1
          Rehabilitation Groups for the 48-Group Model 
            27 RAE
            28 RAD
            29 RAC
            30 RAB
            31 RAA
          Remaining Groups for All Models (66-, 57-, and 48-Groups)  
            32 HE2
            33 HE1
            34 HD2
            35 HD1
            36 HC2
            37 HC1
            38 HB2
            39 HB1
            40 LE2
            41 LE1
            42 LD2
            43 LD1
            44 LC2
            45 LC1
            46 LB2
            47 LB1
            48 CE2
            49 CE1
            50 CD2
            51 CD1
            52 CC2
            53 CC1
            54 CB2
            55 CB1
            56 CA2
            57 CA1
            58 BB2
            59 BB1
            60 BA2
            61 BA1
            62 PE2
            63 PE1
            64 PD2
            65 PD1
            66 PC2
            67 PC1
            68 PB2
            69 PB1
            70 PA2
            71 PA1
            72 AAA

        The AAA group is used when there is no assessment or a valid 
        classification could not be obtained (e.g, if the group obtained 
        for a Medicare Short Stay assessment is not a rehabilitation group).

--------------------------------------------------------------------------- 
RUG-IV RESULTS PRODUCED BY THE CLASSIFICATION LOGIC:

For the selected rehabilitation calculation type (sRehabType), model 
(sModel), and CMIs (nCmiArray), the RUG-IV SAS codes produces normal 
hierarchical classification results, non-therapy hierarchical classification
results, normal index maximizing classification results, and non-therapy 
index maximizing classification results.  The 13 major results variables
are:

   sRugHier (character) - the 3 character RUG group code for normal 
     hierarchical classification.
   nRugHier (numeric) - the numeric RUG group code for normal 
     hierarchical classification.
   nCmiValueHier (numeric) - the CMI value associated with the RUG group
     obtained from normal hierarchical classification.

   sRugMax (character) - the 3 character RUG group code for normal 
     index maximizing classification.
   nRugMax (numeric) - the numeric RUG group code for normal 
     index maximizing classification.
   nCmiValueMax (numeric) - the CMI value associated with the RUG group
     obtained from normal index maximizing classification.

   sRugHier_NT (character) - the 3 character RUG group code for non-therapy 
     hierarchical classification.
   nRugHier_NT (numeric) - the numeric RUG group code for non-therapy 
     hierarchical classification.
   nCmiValueHier_NT (numeric) - the CMI value associated with the RUG group
     obtained from non-therapy hierarchical classification.

   sRugMax_NT (character) - the 3 character RUG group code for non-therapy 
     index maximizing classification.
   nRugMax_NT (numeric) - the numeric RUG group code for non-therapy 
     index maximizing classification.
   nCmiValueMax_NT (numeric) - the CMI value associated with the RUG group
     obtained from non-therapy index maximizing classification.

   sAI_code - Medicare HIPPS assessment indicator (AI) code; last 2 
     charcaters of Medicare HIPPS code

--------------------------------------------------------------------------- 
RUG-IV CATEGORY AND SCALE RESULTS PRODUCED BY THE CLASSIFICATION LOGIC:

     l_Mcare_short_stay - Medicare short stay indicator
     i_adl_tot - total ADL score
     l_depression - depression indicator
     i_rnursing_cnt - count of restorative nursing services (6 days or more)
     i_ext_serv_level - level of extensive services (0 to 3)
     l_sp_high_cond - special care high condition present
     l_sp_low_cond - special care low condition indicator
     l_cog_impair - cognitive impairment indicator
     l_beh_sympt - behavior symptoms indicator 

     l_ext_serv - extensive services category indicator (based on level of
          extensive services and ADL score)
     l_rehab_ultra - ultra high rehab category indicator
     l_rehab_very - very high rehab category indicator
     l_rehab_high - high rehab category indicator
     l_rehab_medium - medium rehab category indicator
     l_rehab_low - low rehab category indicator
     l_special_high - special care high category indicator (based on 
          special care high condition present and ADL score)
     l_special_low - special care low category indicator (based on special
          care low condition present and ADL)
     l_clin_complex - clinically complex category indicator
     l_beh_cog - behavior/cognition category indicator (based on behavior
          symptoms or cognitive impairment indicated) 

--------------------------------------------------------------------------- 
OTHER INTERMEDIATE RESULTS CALCULATED BY THE CLASSIFICATION LOGIC:

     i_sl_skin_tx_cnt - count of special care low skin treatments
     i_cc_skin_tx_cnt - count of clinically complex skin treatments

     n_tot_speech_min - total speech therapy minutes
     n_tot_ot_min - total occupational therapy minutes
     n_tot_pt_min - total physical therapy minutes
     n_tot_rehab_min - total rehab minutes across all three disciplines
     i_rehab_type_5days - number of rehab disciplines received 5 days or more
     i_rehab_type_3days - number of rehab disciplines received 3 days or more
     i_tot_rehab_days - total days of rehab therapy across all 3 disciplines
     n_avg_rehab_min - average minutes of rehab per day across all 3 disciplines

--------------------------------------------------------------------------- 
VERSION NUMBERS SET IN THIS CODE:

     sRugsVersion - RUG-IV logic version and model (includes 1.03 as the logic
          version and the number of groups in the model -- 66, 57, or 48)
     sDllVersion - DLL software version within the RUG-IV logic version
          (always 1.03.0 with this code)
     
--------------------------------------------------------------------------- 
CODE ORGANIZATION:

Version 1.03.0 employs a 2 stage process involving a clinical qualification
  stage followed by a classification stage as follows:
      1.  Clinical Qualification Stage.  Determine all of the RUG-IV clinical
          categories (Extensive/Ultra High Rehab, Extensive/Very High Rehab,
          Extensive/High Rehab, Extensive/Medium Rehab,
          Extensive/Low Rehab, Ultra High Rehab, Very High Rehab,
          High Rehab, Medium Rehab, Low Rehab, Extensive Services,
          Special Care High, Special Care Low, Clinically Complex, and
          Behavior/Cognition) for which a resident qualifies. 
      2.  Classification Stage. Scan all of the qualifications for a resident
          and classify the resident into a RUG-IV group based on hierarchical
          classification, a RUG-IV group based on index maximizing 
          classification, a non-therapy RUG-IV group based on hierarchical 
          classification, a non-therapy RUG-IV group based on index maximizing
          classification.

---------------------------------------------------------------------------
RUG-IV PRIOR VERSION HISTORY:
   Version 1.00  SAS Code          10/20/09 by BG - initial version
   Version 1.00.1 SAS Code         11/27/09 by BG - revision
   Version 1.00.2 SAS Code         11/30/09 by BG - revision
   Version 1.00.3 SAS Code         12/03/09 by BG - revision
   Version 1.00.4 SAS Code         12/14/09 by BG - revision
                                     -revised SOT and add SOT error code
                                     -added K0500A as qualifer for 
                                      Special Care High
   Version 1.00.5 SAS Code         05/19/10 by BG - revision
                                     -corrected use of K0300 as a qualifier
                                      for Special Care High
                                     -corrected the coding for the iError = 5
                                      condition which occurs when a Start of
                                      Therapy OMRA produces an index
                                      maximized RUG-IV classification below the
                                      Rehabilitation Plus Extensive and 
                                      Rehabilitation groups 
   Version 1.00.6 SAS Code         06/28/10 by BG - revision
                                     -Cognitive impairment: added logic to
                                      use the staff assessment for mental status
                                      when B0500 = '-'.
                                     -Cognitive impairment: corrected the coding
                                      for the staff assessment determination of
                                      impairment to indicate impairment when there
                                      are two or more impairment indicators AND
                                      there is one or more severe impairment 
                                      indicators (was "OR" in previous version).
                                     -Cognitive impairment: corrected the coding
                                      for count of impairment indicators to 
                                      increment count when 
                                         (C1000 > '0') AND (C1000 <= '3')
                                      In the previous version the second part
                                      of this phrase was incorrectly coded as
                                         (C1000 > '0') AND (B0700 <= '3') 
                                     -Input parameter error: added code to set
                                      iError = 4 condition if any CMI value 
                                      input in nCmiArray is > 9999.                                    
   Version 1.00.7 SAS Code         07/27/10 by BG - revision (NOT PUBLIC)
                                     -The multiplier used in the rehabilitation
                                      therapy group time adjustment (when group 
                                      therapy represents more than 25% of total 
                                      minutes) was revised from 1.33 to 4.0/3.0.
                                     -The comment describing the second condition
                                      required for indentifying a Medicare Short
                                      Stay Assessment has been clarified.  Although
                                      this is a required condition, it is noted 
                                      that it can only be tested with consideration
                                      of the prior MDS 3.0 record history.  The 
                                      RUG-IV SAS code only accesses a single record
                                      at a time and cannot test this condition.
   Version 1.00.8 SAS Code         08/17/10 by BG - revision 
                                     -For a start of therapy OMRA combined with 
                                      an end of therapy OMRA, the non-therapy 
                                      classification was not reset to the AAA 
                                      default group as was erroneously done in 
                                      previous versions.
                                     -When RUG-IV group classification results were
                                      reset to the AAA default group for a start of 
                                      therapy OMRA, the corresponding non-therapy
                                      CMI results were also reset to the default
                                      value of 0.0.
   Version 1.00.9 SAS Code         10/08/10 by BG - revision
                                     -Changes were made to the logic for Start of
                                      Therapy (SOT) OMRAs to properly handle their 
                                      usage for non-Medicare purposes.  

                                      If an assessment is a Start of Therapy (SOT) 
                                      OMRA (whether or not it is combined with
                                      other types of assessments), then the
                                      Medicare RUG-IV classification must be in a
                                      group involving rehabilitation
                                      (Rehabilitation Plus Extensive Services group
                                      or a Rehabilitation group).  A Medicare 
                                      non-rehabilitation classification is not
                                      allowed for an SOT OMRA and the CMS
                                      Assessment Submission and Processing (ASAP) 
                                      system will reject the assessment if a
                                      Medicare non-rehabilitation classification
                                      occurs.  In previous RUG-IV versions, this
                                      requirement was incorrectly applied to both
                                      Medicare classification and non-Medicare
                                      classification.  This requirement does not
                                      apply to non-Medicare classification. 

                                      RUG-IV Version 1.00.9 corrected the handling
                                      of SOT OMRAs with the following logic:

                                      1) For Medicare classification (sRehabType = 
                                         "Mcare") on a SOT OMRA:
                                         a) If the index maximized classification 
                                            (sRugMax) is a group not involving 
                                            rehabilitation (group does not start
                                            with "R"), then set all RUG group
                                            results to the default AAA group and
                                            set the grouper error code (iError) to
                                            5 indicating an invalid Medicare
                                            classification.
                                         b) If the SOT OMRA is not combined with an
                                            OBRA assessment (excluding a discharge
                                            assessment) or another PPS assessment,
                                            then many items required to make a 
                                            non-therapy classification are not
                                            present and set the non-therapy RUG 
                                            group results (sRugHier_NT and
                                            sRugMax_NT) to the default AAA group.
                                            No error code is set.

                                      2) For non-Medicare classification
                                         (sRehabType = "Other") on a SOT OMRA:
                                         a) If the SOT OMRA is not combined with an
                                            OBRA assessment (excluding a discharge
                                            assessment) or another PPS assessment,
                                            then many items required to make a 
                                            RUG-IV classification are not present
                                            and set the all RUG group results 
                                            to the default AAA group.  No error
                                            code is set.
   Version 1.01.1 SAS Code        7/12/11 by BG - revision
                                     -Add a new value to rehabilitation 
                                      classification type parameter (sRehabType).
                                      This new value of MCAR2 works the same as
                                      the MCARE value but adds allocation of
                                      group therapy time for rehabilitation
                                      classification.
                                    - Add new Assessment Indicator (AI) codes for
                                      assessments involving resumption and change
                                      in rehabilitation therapy.  New MDS 3.0 item
                                      O0450A Resumption of Therapy Code used.
                                    - Fix a very rare rounding error that occurred
                                      in the calculation of total rehabilitation
                                      minutes on some platforms.
                                    - Correct the Medicare Short Stay Assessment
                                      indicator code.  In previous versions, the
                                      Short Stay indicator could be set if the
                                      assessment was a Start of Therapy OMRA
                                      whether or not combined with an End of
                                      Therapy OMRA.  The code has been corrected
                                      to set the Short Stay indicator only if the
                                      assessment is a Start of Therapy OMRA NOT
                                      combined with an End of Therapy OMRA.
                                    - Change the code for C0500 (BIMS summary
                                      score), D0300 (PHQ resident interview
                                      summary score), and D0600 (PHQ staff
                                      assessment summary score) to insure proper
                                      handling of numeric values that are 
                                      right-justified with leading-zero fill 
                                      or are left-justified.
                                    - Change the Medicare 25% group therapy
                                      limitation code to avoid SAS warning
                                      messages when therapy minutes items have
                                      dash ('-') values.
                                    - When a Start of Therapy OMRA produces a
                                      non-therapy classification, then reset the
                                      resulting CMI value to the value for the
                                      default group, since classification in a
                                      non-therapy group is invalid.
   Version 1.02.0 SAS Code        1/4/12 by BG - revision
                                   - For assessments with assessment reference 
                                     date (A2300) on or after 4/1/2012, K0500A 
                                     is being replaced by the pair of items 
                                     K0510A1 and K0510A2 and K0500B is being 
                                     replaced by the pair of items K0510B1 and
                                     K0510B2.
   Version 1.02.1 SAS Code        7/24/12 by BG - revision
                                  - This version corrects a coding error in the 
                                    DLL.  This coding error did not occur in the
                                    SAS code.  The only change to the SAS code 
                                    was to set the DLL version code (sDllVersion) 
                                    to 1.02.1 (the new Code Version code).

---------------------------------------------------------------------------

=========================END OF DOCUMENTATION==============================
*/

/* INITIALIZE RESULTS VARIABLES
*/
   sRugHier = 'AAA';
   nRugHier = 72;
   nCmiValueHier = 0.0;

   sRugMax = 'AAA';
   nRugMax = 72;
   nCmiValueMax = 0.0;

   sRugHier_NT = 'AAA';
   nRugHier_NT = 72;
   nCmiValueHier_NT = 0.0;

   sRugMax_NT = 'AAA';
   nRugMax_NT = 72;
   nCmiValueMax_NT = 0.0;

   sAI_code = '  ';

   l_Mcare_short_stay = 0;
   i_adl_tot = 0;
   l_depression = 0;
   i_rnursing_cnt = 0;
   i_ext_serv_level = 0;
   l_sp_high_cond = 0;
   l_sp_low_cond = 0;
   l_cog_impair = 0;
   l_beh_sympt = 0;

   l_ext_serv = 0;
   l_rehab_ultra = 0;
   l_rehab_very = 0;
   l_rehab_high = 0;
   l_rehab_medium = 0;
   l_rehab_low = 0;
   l_special_high = 0;
   l_special_low = 0;
   l_clin_complex = 0;
   l_beh_cog = 0; 

   i_sl_skin_tx_cnt = 0;
   i_cc_skin_tx_cnt = 0;

   n_tot_speech_min = 0.0;
   n_tot_ot_min = 0.0;
   n_tot_pt_min = 0.0;
   n_tot_rehab_min = 0.0;
   i_rehab_type_5days = 0;
   i_rehab_type_3days = 0;
   i_tot_rehab_days = 0;
   n_avg_rehab_min = 0;

/* SET VERSION CODES (add num gps later if model valid)
*/
*First part of RUGs version code;
sRugsVersion = '1.03';
*V1.03.0 CHANGE to DLL version code;
sDllVersion = '1.03.0';

/* CHECK PARAMETERS AND SET ERROR CODE (iError).
*/
iError = 0;
sRehabType = UPCASE(sRehabType);
*V1.03.0 CHANGE to include new sRehabType = 'MCAR3';
IF sRehabType NOT = 'MCARE' AND sRehabType NOT = 'MCAR2' AND 
     sRehabType NOT = 'MCAR3' AND sRehabType NOT = 'OTHER' 
          THEN DO;
               iError = 1;
               GO TO SKIPEND;
          END;

IF sModel NOT = '48' AND sModel NOT = '57' AND sModel NOT = '66' THEN 
   DO;
      iError = 2;
      GO TO SKIPEND;
   END;
*Complete sRugsVersion if sModel is valid;
ELSE sRugsVersion = sRugsVersion || sModel;

IF A0310A = '99' AND A0310B = '99' THEN
   DO;
      sAI_code = 'X ';
      iError = 3;
      GO TO SKIPEND;
   END;

IF (A0310A < '01') OR (A0310A > '06' AND A0310A NOT = '99') THEN
   DO;
      iError = 3;
      GO TO SKIPEND;
   END;
* Abt 4/10/2020 - changed 07 to 08 for A0310B;
IF (A0310B < '01') OR (A0310B > '08' AND A0310B NOT = '99') THEN
   DO;
      iError = 3;
      GO TO SKIPEND;
   END;
* Abt 4/10/2020 - allowing missing on A0310C (may want to refine this);
IF ((A0310C < '0') OR (A0310C > '4')) AND (A0310C NOT = ' ') THEN
   DO;
      iError = 3;
      GO TO SKIPEND;
   END;
* Abt 4/10/2020 - allowing missing on A0310D (may want to refine this);
IF (A0310D NOT = '^') AND (A0310D NOT = '0') AND (A0310D NOT = '1') AND (A0310D NOT = ' ') THEN
   DO;
      iError = 3;
      GO TO SKIPEND;
   END;

*Check CMI array for invalid values and quit if invalid value found          ;
DO i = 1 to 72;
   IF nCmiArray(i) <= -9999 THEN iError = 4;
   IF nCmiArray(i) > 9999 THEN iError = 4;
END;
IF iError = 4 THEN 
      GO TO SKIPEND;

/*
========================================
BEGIN RUG-IV CALCULATION. 
========================================
*/

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
CALCULATION OF TOTAL ADL SCORE.                                           
     The Activities of Daily Living (ADL) score is used in all determinations
     of a resident's placement in a RUG-IV group. The ADL score is based upon 
     the four "late loss" ADLs (bed mobility, transfer, toilet use and 
     eating), and this score indicates the level of functional assistance or 
     support required by the resident.  It is a very important component of 
     the classification process.
*/

/*---------------------------------------------------------------------------
STEP #1
     ADL scoring conversion for bed mobility (G0110B), 
     transfer (G0110B), and toilet use (G0110I).  These 3 ADLs all use 
     parallel logic.
*/

*Bed mobility (G0110A) scoring conversion;
i_bedmob = 0; 
IF      (G0110A1 = '-' OR G0110A1 = '0' OR G0110A1 = '1' OR G0110A1 = '7' OR 
         G0110A1 = '8') 
             THEN i_bedmob = 0;
ELSE IF (G0110A1 = '2') 
             THEN i_bedmob = 1;
ELSE IF (G0110A1 = '3') AND
        (G0110A2 = '-' OR G0110A2 = '0' OR G0110A2 = '1' OR G0110A2 = '2')
             THEN i_bedmob = 2;
ELSE IF (G0110A1 = '4') AND
        (G0110A2 = '-' OR G0110A2 = '0' OR G0110A2 = '1' OR G0110A2 = '2')
             THEN i_bedmob = 3;
ELSE IF (G0110A1 = '3' OR G0110A1 = '4') AND
        (G0110A2 = '3')
             THEN i_bedmob = 4;

*Transfer (G0110B) scoring conversion;
i_transfer = 0;
IF      (G0110B1 = '-' OR G0110B1 = '0' OR G0110B1 = '1' OR G0110B1 = '7' OR 
         G0110B1 = '8') 
             THEN i_transfer = 0;
ELSE IF (G0110B1 = '2') 
             THEN i_transfer = 1;
ELSE IF (G0110B1 = '3') AND
        (G0110B2 = '-' OR G0110B2 = '0' OR G0110B2 = '1' OR G0110B2 = '2')
             THEN i_transfer = 2;
ELSE IF (G0110B1 = '4') AND
        (G0110B2 = '-' OR G0110B2 = '0' OR G0110B2 = '1' OR G0110B2 = '2')
             THEN i_transfer = 3;
ELSE IF (G0110B1 = '3' OR G0110B1 = '4') AND
        (G0110B2 = '3')
             THEN i_transfer = 4;

*Toilet use (G0110I) scoring conversion;
i_toilet = 0;
IF      (G0110I1 = '-' OR G0110I1 = '0' OR G0110I1 = '1' OR G0110I1 = '7' OR 
         G0110I1 = '8') 
             THEN i_toilet = 0;
ELSE IF (G0110I1 = '2') 
             THEN i_toilet = 1;
ELSE IF (G0110I1 = '3') AND
        (G0110I2 = '-' OR G0110I2 = '0' OR G0110I2 = '1' OR G0110I2 = '2')
             THEN i_toilet = 2;
ELSE IF (G0110I1 = '4') AND
        (G0110I2 = '-' OR G0110I2 = '0' OR G0110I2 = '1' OR G0110I2 = '2')
             THEN i_toilet = 3;
ELSE IF (G0110I1 = '3' OR G0110I1 = '4') AND
        (G0110I2 = '3')
             THEN i_toilet = 4;

/*---------------------------------------------------------------------------
STEP #2
     ADL scoring conversion for eating(G0110H). 
*/
i_eating = 0;
IF      (G0110H1 = '-' OR G0110H1 = '0' OR G0110H1 = '1' OR G0110H1 = '2' OR 
         G0110H1 = '7' OR G0110H1 = '8') AND
        (G0110H2 = '-' OR G0110H2 = '0' OR G0110H2 = '1' OR G0110H2 = '8')
             THEN i_eating = 0;
IF      (G0110H1 = '-' OR G0110H1 = '0' OR G0110H1 = '1' OR G0110H1 = '2' OR 
         G0110H1 = '7' OR G0110H1 = '8') AND
        (G0110H2 = '2' OR G0110H2 = '3')
             THEN i_eating = 2;
IF      (G0110H1 = '3') AND 
        (G0110H2 = '-' OR G0110H2 = '0' OR G0110H2 = '1')
             THEN i_eating = 2;
IF      (G0110H1 = '3') AND 
        (G0110H2 = '2' OR G0110H2 = '3')
             THEN i_eating = 3;
IF      (G0110H1 = '4') AND 
        (G0110H2 = '-' OR G0110H2 = '0' OR G0110H2 = '1')
             THEN i_eating = 2;
IF      (G0110H1 = '4') AND 
        (G0110H2 = '2' OR G0110H2 = '3')
             THEN i_eating = 4;

/*---------------------------------------------------------------------------
STEP #3
     Add the four scores for the total ADL score.  This is the RUG-IV TOTAL 
     ADL SCORE.  The total ADL score ranges from 0 through 16. 
*/
   i_adl_tot = i_bedmob + i_transfer + i_toilet + i_eating;

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
CALCULATION OF TOTAL REHABILITATION THERAPY MINUTES

For Speech-Language Pathology Services (Items at O0400A), Occupational 
Therapy (Items at O0400B), and Physical Therapy (Items at O0400C), the 
MDS 3.0 separately captures minutes that the resident was in individual 
therapy, in concurrent therapy (therapy with one other patient), and in 
group therapy (part of a group of 4 residents) during the last 7 days.
For each therapy discipline, total minutes are calculated as follows:
   1. TOTAL MINUTES:  
      a. For FY2011 Special Medicare rehabilitation classification
         (sRehabType = 'MCARE') and for non-Medicare rehabilitation
         classification (sRehabType = 'OTHER'), TOTAL MINUTES includes all
         minutes in individual therapy, one-half of the minutes in 
         concurrent therapy, and all minutes in group therapy.  

      b. For FY2012 Special Medicare rehabilitation classification
         (sRehabType = 'MCAR2')and for FY2014 Special Medicare 
         rehabilitation classification (sRehabType = 'MCAR3'), TOTAL 
         MINUTES includes all minutes in individual therapy, one-half of 
         the minutes in concurrent therapy, and one-quarter of the 
         minutes in group therapy. 

         NOTE: V1.03.0 CHANGES this condition to include new sRehabType 
         = 'MCAR3';
 
   2. ADJUSTMENT OF TOTAL MINUTES.  
      For Medicare Part A (sRehabType = 'MCARE', 'MCAR2', or 'MCAR3') 
      there is a limitation that the group minutes cannot exceed 25% of 
      the TOTAL MINUTES.  For Medicare Part A, ADJUSTED TOTAL MINUTES 
      includes all minutes in individual therapy, one-half of the minutes 
      in concurrent therapy, and minutes in group therapy up to a number 
      of minutes equal to 25% of the final ADJUSTED TOTAL MINUTES. If the 
      group therapy minutes exceeds 25% of the unadjusted TOTAL MINUTES, 
      then the ADJUSTED TOTAL MINUTES equals 4/3 times the total 
      individual minutes plus 4/3  one-half times concurrent minutes.

      NOTE: V1.03.0 CHANGES this condition to include new sRehabType 
      = 'MCAR3'.

      The group therapy limitation may also be used for other payment 
      systems. If the group adjustment is used, then RUG-IV therapy
      classification is based on the ADJUSTED TOTAL MINUTES.  If the 
      group adjustment is not used, then RUG-IV therapy classification is
      based on the unadjusted TOTAL MINUTES.
*/

/*---------------------------------------------------------------------------
STEP #1
Calculate the total minutes (n_tot_speech_min) for speech-language pathology
   services.
*/

/*
Calculate unadjusted total minutes.  Only consider valid numeric values
   (ignore non-assessed dash (-) and skipped blank (^) items).
*/
n_tot_speech_min = 0.0;
IF ( (O0400A1 >= '0001') AND (O0400A1 <= '9999') )
     THEN n_tot_speech_min = n_tot_speech_min + INPUT(O0400A1, 4.);
IF ( (O0400A2 >= '0001') AND (O0400A2 <= '9999') )
     THEN n_tot_speech_min = n_tot_speech_min 
               + INPUT(O0400A2, 4.) / 2;
IF ( (O0400A3 >= '0001') AND (O0400A3 <= '9999') )
     THEN DO;
          *V1.03.0 CHANGE to include new sRehabType = 'MCAR3';
          IF (sRehabType = 'MCAR2' OR sRehabType = 'MCAR3')
               THEN n_tot_speech_min = n_tot_speech_min
                    + INPUT(O0400A3, 4.) / 4.;
          ELSE n_tot_speech_min = n_tot_speech_min 
                    + INPUT(O0400A3, 4.);
     END;

/*Apply group time adjustment if the rehabilitation calculation type is 
   'MCARE', 'MCAR2' or 'MCAR3'.
*/
IF ( (O0400A3 >= '0001') AND (O0400A3 <= '9999') )
     THEN DO;
          *V1.03.0 CHANGE to include new sRehabType = 'MCAR3';
          IF (sRehabType = 'MCAR2' OR sRehabType = 'MCAR3') AND 
             (n_tot_speech_min > 0) AND 
             ((INPUT(O0400A3, 4.) / 4.) / n_tot_speech_min > 0.25) 
                THEN DO;
                     n_tot_speech_min = 0;
                     IF ( (O0400A1 >= '0001') AND (O0400A1 <= '9999') )
                         THEN n_tot_speech_min = n_tot_speech_min 
                                  + (INPUT(O0400A1,4.)) * 4.0 / 3.0;
                     IF ( (O0400A2 >= '0001') AND (O0400A2 <= '9999') )
                         THEN n_tot_speech_min = n_tot_speech_min 
                                  + (INPUT(O0400A2,4.) / 2.0) * 4.0 / 3.0;
                END;
          ELSE IF (sRehabType = 'MCARE') AND 
             (n_tot_speech_min > 0) AND 
             ((INPUT(O0400A3, 4.)) / n_tot_speech_min > 0.25) 
                THEN DO;
                     n_tot_speech_min = 0;
                     IF ( (O0400A1 >= '0001') AND (O0400A1 <= '9999') )
                         THEN n_tot_speech_min = n_tot_speech_min 
                                  + (INPUT(O0400A1,4.)) * 4.0 / 3.0;
                     IF ( (O0400A2 >= '0001') AND (O0400A2 <= '9999') )
                         THEN n_tot_speech_min = n_tot_speech_min 
                                  + (INPUT(O0400A2,4.) / 2.0) * 4.0 / 3.0;
                END;
      END;

/*---------------------------------------------------------------------------
STEP #2
Calculate the total minutes (n_tot_ot_min) for occupational therapy
   services.
*/

/*
Calculate unadjusted total minutes.  Only consider valid numeric values
(ignore non-assessed dash (-) and skipped blank (^) items).
*/
n_tot_ot_min = 0.0;
IF ( (O0400B1 >= '0001') AND (O0400B1 <= '9999') )
     THEN n_tot_ot_min = n_tot_ot_min + INPUT(O0400B1, 4.);
IF ( (O0400B2 >= '0001') AND (O0400B2 <= '9999') )
     THEN n_tot_ot_min = n_tot_ot_min 
               + INPUT(O0400B2, 4.) / 2;
IF ( (O0400B3 >= '0001') AND (O0400B3 <= '9999') )
     THEN DO;
          *V1.03.0 CHANGE to include new sRehabType = 'MCAR3';
          IF (sRehabType = 'MCAR2' OR sRehabType = 'MCAR3')
               THEN n_tot_ot_min = n_tot_ot_min
                    + INPUT(O0400B3, 4.) / 4.;
          ELSE n_tot_ot_min = n_tot_ot_min 
                    + INPUT(O0400B3, 4.);
     END;

/*Apply group time adjustment if the rehabilitation calculation type is 
   'MCARE', 'MCAR2', or 'MCAR3'.
*/
IF ( (O0400B3 >= '0001') AND (O0400B3 <= '9999') )
     THEN DO;
          *V1.03.0 CHANGE to include new sRehabType = 'MCAR3';
          IF (sRehabType = 'MCAR2' OR sRehabType = 'MCAR3') AND 
             (n_tot_ot_min > 0) AND 
             ((INPUT(O0400B3, 4.) / 4.) / n_tot_ot_min > 0.25) 
                THEN DO;
                     n_tot_ot_min = 0;
                     IF ( (O0400B1 >= '0001') AND (O0400B1 <= '9999') )
                         THEN n_tot_ot_min = n_tot_ot_min 
                                  + (INPUT(O0400B1,4.)) * 4.0 / 3.0;
                     IF ( (O0400B2 >= '0001') AND (O0400B2 <= '9999') )
                         THEN n_tot_ot_min = n_tot_ot_min 
                                  + (INPUT(O0400B2,4.) / 2.0) * 4.0 / 3.0;
                END;
          ELSE IF (sRehabType = 'MCARE') AND 
             (n_tot_ot_min > 0) AND 
             ((INPUT(O0400B3, 4.)) / n_tot_ot_min > 0.25) 
                THEN DO;
                     n_tot_ot_min = 0;
                     IF ( (O0400B1 >= '0001') AND (O0400B1 <= '9999') )
                         THEN n_tot_ot_min = n_tot_ot_min 
                                  + (INPUT(O0400B1,4.)) * 4.0 / 3.0;
                     IF ( (O0400B2 >= '0001') AND (O0400B2 <= '9999') )
                         THEN n_tot_ot_min = n_tot_ot_min 
                                  + (INPUT(O0400B2,4.) / 2.0) * 4.0 / 3.0;
                END;
      END;

/*---------------------------------------------------------------------------
STEP #3
Calculate the total minutes (n_tot_pt_min) for physical therapy
   services.
*/

/*
Calculate unadjusted total minutes.  Only consider valid numeric values
(ignore non-assessed dash (-) and skipped blank (^) items).
*/
n_tot_pt_min = 0.0;
IF ( (O0400C1 >= '0001') AND (O0400C1 <= '9999') )
     THEN n_tot_pt_min = n_tot_pt_min + INPUT(O0400C1, 4.);
IF ( (O0400C2 >= '0001') AND (O0400C2 <= '9999') )
     THEN n_tot_pt_min = n_tot_pt_min 
               + INPUT(O0400C2, 4.) / 2;
IF ( (O0400C3 >= '0001') AND (O0400C3 <= '9999') )
     THEN DO;
          *V1.03.0 CHANGE to include new sRehabType = 'MCAR3';
          IF (sRehabType = 'MCAR2' OR sRehabType = 'MCAR3')
               THEN n_tot_pt_min = n_tot_pt_min
                    + INPUT(O0400C3, 4.) / 4.;
          ELSE n_tot_pt_min = n_tot_pt_min 
                    + INPUT(O0400C3, 4.);
     END;

/*Apply group time adjustment if the rehabilitation calculation type is 
   'MCARE', 'MCAR2', or 'MCAR3'.
*/
IF ( (O0400C3 >= '0001') AND (O0400C3 <= '9999') )
     THEN DO;
          *V1.03.0 CHANGE to include new sRehabType = 'MCAR3';
          IF (sRehabType = 'MCAR2' OR sRehabType = 'MCAR3') AND 
             (n_tot_pt_min > 0) AND 
             ((INPUT(O0400C3, 4.) / 4.) / n_tot_pt_min > 0.25) 
                THEN DO;
                     n_tot_pt_min = 0;
                     IF ( (O0400C1 >= '0001') AND (O0400C1 <= '9999') )
                         THEN n_tot_pt_min = n_tot_pt_min 
                                  + (INPUT(O0400C1,4.)) * 4.0 / 3.0;
                     IF ( (O0400C2 >= '0001') AND (O0400C2 <= '9999') )
                         THEN n_tot_pt_min = n_tot_pt_min 
                                  + (INPUT(O0400C2,4.) / 2.0) * 4.0 / 3.0;
                END;
          ELSE IF (sRehabType = 'MCARE') AND 
             (n_tot_pt_min > 0) AND 
             ((INPUT(O0400C3, 4.)) / n_tot_pt_min > 0.25) 
                THEN DO;
                     n_tot_pt_min = 0;
                     IF ( (O0400C1 >= '0001') AND (O0400C1 <= '9999') )
                         THEN n_tot_pt_min = n_tot_pt_min 
                                  + (INPUT(O0400C1,4.)) * 4.0 / 3.0;
                     IF ( (O0400C2 >= '0001') AND (O0400C2 <= '9999') )
                         THEN n_tot_pt_min = n_tot_pt_min 
                                  + (INPUT(O0400C2,4.) / 2.0) * 4.0 / 3.0;
                END;
      END;

/*---------------------------------------------------------------------------
STEP #4
Calculate the sum of TOTAL MINUTES (n_tot_rehab_min) across all three 
   therapy disciplines.
NOTE:  To correct a rare rounding issue on some platforms, force
       rounding of the fourth decimal.  The rounding method used is 
       "round-half-up (arithmetic rounding)".  With this method, the fourth
       decimal place is rounded down if the fifth place is 1 through 4 and
       rounded up if the fifth place is 5 through 9.

NOTE:  The "round-half-up (arithmetic rounding)" method must be used
       to match the test data provided with the RUG-IV V1.03.0
       package.  
*/

n_tot_rehab_min = n_tot_speech_min + n_tot_ot_min + n_tot_pt_min; 
n_tot_rehab_min = ROUND(n_tot_rehab_min, 0.0001);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
CALCULATION OF REHABILITATION THERAPY DAYS VARIABLES

Three different variables based on the number of rehabilitation therapy
days are needed for classification:
1. i_rehab_types_5days = number of therapy disciplines (speech, OT, and PT)
   received for 5 or more days.
2. i_rehab_types_3days = number of therapy disciplines (speech, OT, and PT)
   received for 3 or more days.
3. i_tot_rehab_days = total number of days of therapy across the therapy
   disciplines.  
*/

i_rehab_type_5days = 0;
IF ( (O0400A4 >= '5') AND (O0400A4 <= '7') )
     THEN i_rehab_type_5days = i_rehab_type_5days + 1; 
IF ( (O0400B4 >= '5') AND (O0400B4 <= '7') )
     THEN i_rehab_type_5days = i_rehab_type_5days + 1; 
IF ( (O0400C4 >= '5') AND (O0400C4 <= '7') )
     THEN i_rehab_type_5days = i_rehab_type_5days + 1; 

i_rehab_type_3days = 0;
IF ( (O0400A4 >= '3') AND (O0400A4 <= '7') )
     THEN i_rehab_type_3days = i_rehab_type_3days + 1; 
IF ( (O0400B4 >= '3') AND (O0400B4 <= '7') )
     THEN i_rehab_type_3days = i_rehab_type_3days + 1; 
IF ( (O0400C4 >= '3') AND (O0400C4 <= '7') )
     THEN i_rehab_type_3days = i_rehab_type_3days + 1; 

i_tot_rehab_days = 0;
IF ( (O0400A4 >= '1') AND (O0400A4 <= '7') )
     THEN i_tot_rehab_days = i_tot_rehab_days + INPUT(O0400A4, 1.);
IF ( (O0400B4 >= '1') AND (O0400B4 <= '7') )
     THEN i_tot_rehab_days = i_tot_rehab_days + INPUT(O0400B4, 1.);
IF ( (O0400C4 >= '1') AND (O0400C4 <= '7') )
     THEN i_tot_rehab_days = i_tot_rehab_days + INPUT(O0400C4, 1.);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
MEDICARE SHORT STAY ASSESSMENT HANDLING

RUG-IV uses an alternative rehabilitation therapy classification when an
assessment is a Medicare Short Stay assessment.  A Medicare Short Stay 
assessment is a Start of Therapy OMRA assessment that satisfies several 
other conditions.  To be considered as a Medicare Special Short Stay 
assessment and use the special RUG-IV short stay rehabilitation therapy 
classification, all of the 7 following conditions must be met:
   1. The assessment must be a Start of Therapy OMRA A0310C = 1) NOT 
      combined with an End of Therapy OMRA.  However, this assessment may 
      be combined with any OBRA assessment, any scheduled PPS assessment, 
      and/or a Swing Bed clinical change assessment.
   2. The Start of Therapy OMRA must be combined with a PPS 5-day
      (A0310B = 01) or readmission/return assessment (A0310B = 06), or a
      PPS 5-day or re-admission/return assessment must have been performed
      earlier in the short stay.  The Start of Therapy OMRA should also be
      combined with a discharge assessment when the resident is discharged
      from the facility, but not combined with a discharge if the resident
      dies in the facility or is transferred to Long-Term Care in the facility.
      NOTE: 
         According to Medicare SNF PPS assessment requirements, a stand-alone 
         Start of Therapy OMRA should never be performed before the 5-Day 
         or Readmission/Return assessment for a Medicare stay.  If a Start 
         of Therapy OMRA is performed and the 5-Day or Readmission/Return 
         assessment has not yet been performed, then the following Medicare 
         SNF PPS requirements apply:
         a. That unscheduled Start of Therapy OMRA assessment replaces the 
            scheduled 5-Day or Readmission/Return assessment.  A subsequent 
            5-Day or Readmission/Return assessment is not allowed.
         b. The assessment should be coded as both a Start of Therapy OMRA 
            and a 5-Day or Readmission/Return assessment.
         c. All requirements for both the Start of Therapy OMRA and the a 
            5-Day or Readmission/Return assessment must be satisfied. 
         It is the facility's responsibility to insure compliance with these 
         requirements.  Any Start of Therapy OMRA assessment that complies 
         with these requirements will automatically satisfy the Medicare Short 
         Stay Assessment condition #2 above.  It is not necessary that the 
         standard RUG-IV grouper (DLL or SAS code) actually test the second 
         condition, since it should always be true.  Private software vendors, 
         who develop their own RUG-IV classification code, need not test the 
         second condition when classifying a Start of Therapy OMRA assessment 
         as a Medicare Short Stay assessment.  However, software vendors may 
         want to alert the facility when a Start of Therapy OMRA precedes the 
         5-Day or Readmission/Return assessment.  The facility should be 
         advised that this is not allowed and the facility must combine the 
         Start of Therapy OMRA assessment with the scheduled 5-Day or 
         Readmission/Return assessment.
   3. The assessment reference date (A2300) must be on the 8th day of a Part
      A Medicare stay or earlier in that stay.  That is, the assessment 
      reference date minus the start of Medicare stay date (A2400B) must 
      be 7 days or less.
   4. The Medicare Part A covered stay must end on the assessment reference
      date (A2300) of the Start of Therapy OMRA.  That assessment reference
      date must equal the end of Medicare stay date (A2400C).  
   5. Rehabilitation therapy (speech-language pathology services, or 
      occupational or physical therapy) has started during the last 4 days
      of the Medicare Part A covered stay.  That is, the end of Medicare
      stay date (A2400C) minus the earliest start date for these 3 therapy
      disciplines (O0400A5, O0400B5, or O0400C5) must be 3 days or less.
   6. Rehabilitation therapy must not have ended before the last day of the
      Medicare Part A covered stay.  That is, at least one of the therapy
      disciplines must have a dash-filled end of therapy date (O0400A6,
      O0400B6, or O0400C6) indicating on-going therapy or an end of therapy
      date equal to the end of covered Medicare stay date (A2400C).
   7. The rehabilitation calculation type is Medicare (sRehabType =
      'MCARE', 'MCAR2', or 'MCAR3').

          NOTE: 'MCAR3' added to this condition with Version 1.03.0. 

If all 7 conditions are satisfied, set the Medicare Short Stay Assessment
Indicator (l_Mcare_short_stay) to 1 (yes), otherwise set the indicator to 0 
(no).  If indicator = 1, then calculate the average therapy minutes per day. 
*/

/*---------------------------------------------------------------------------
STEP #1
Set the Medicare Short Stay Assessment Indicator. 
*/

/*
Screen dates and set bad date indicator (l_bad_date) = 1 if dates preclude
identification of Medicare Short Stay. Convert valid dates to SAS date 
variables. 
*/
l_bad_date = 0;

*Handle assessment reference date A2300;
i_month  = INPUT(SUBSTR(A2300,5,2), 2.);    
i_day    = INPUT(SUBSTR(A2300,7,2), 2.);    
i_year   = INPUT(SUBSTR(A2300,1,4), 4.);    
d_A2300  = MDY(i_month, i_day, i_year);
*If d_A2300 is an invalid date then it will have a missing value indicated in 
    SAS by a period (.);
IF (d_A2300 = .) THEN l_bad_date = 1;

*Handle start of Medicare stay date A2400B;
IF l_bad_date = 0
     THEN DO;
          IF (SUBSTR(A2400B,1,1) = '^')
               THEN l_bad_date = 1;
          ELSE DO; 
               i_month  = INPUT(SUBSTR(A2400B,5,2), 2.);    
               i_day    = INPUT(SUBSTR(A2400B,7,2), 2.);    
               i_year   = INPUT(SUBSTR(A2400B,1,4), 4.);    
               d_A2400B = MDY(i_month, i_day, i_year);
               *If d_A2400B is an invalid date then it will have a missing 
                   value indicated in SAS by a period (.);
               IF (d_A2400B = .) THEN l_bad_date = 1;
          END;
     END;

/*
Handle end of Medicare stay date A2400C.  Set bad date indicator if no end 
of Medicare stay date, the date is dash-filled indicating that the stay 
is on-going, or the date contains a single dash indicating the item was not
assessed.
*/
IF l_bad_date = 0
     THEN DO;
          IF ( (SUBSTR(A2400C,1,1) = '^') OR 
               (SUBSTR(A2400C,1,1) = '-') OR
               (A2400C = '--------') )  
                    THEN l_bad_date = 1;
          ELSE DO; 
               i_month  = INPUT(SUBSTR(A2400C,5,2), 2.);    
               i_day    = INPUT(SUBSTR(A2400C,7,2), 2.);    
               i_year   = INPUT(SUBSTR(A2400C,1,4), 4.);    
               d_A2400C = MDY(i_month, i_day, i_year);
               *If d_A2400C is an invalid date then it will have a missing 
                   value indicated in SAS by a period (.);
               IF (d_A2400C = .) THEN l_bad_date = 1;
          END;
     END;

/*
Handle start of therapy dates O0400A5, O0400B5, and O0400C5.
Determine earliest start of rehab therapy date (d_rehab_start) and set bad 
date indicator (l_bad_date = 1) if no start of rehab therapy date. 
*/
IF l_bad_date = 0
     THEN DO;
          *Initialize rehab therapy start date to future date for comparison;
          d_rehab_start = MDY(1, 1, 2099);

          *Check for no start of rehab therapy date;
          IF  ( (SUBSTR(O0400A5,1,1) = '^' OR SUBSTR(O0400A5,1,1) = '-') AND
                (SUBSTR(O0400B5,1,1) = '^' OR SUBSTR(O0400B5,1,1) = '-') AND
                (SUBSTR(O0400C5,1,1) = '^' OR SUBSTR(O0400C5,1,1) = '-') )
                     THEN l_bad_date = 1;
          ELSE DO;
               *If speech therapy start date (O0400A5) present;
               IF (SUBSTR(O0400A5,1,1) NOT = '^' AND
                   SUBSTR(O0400A5,1,1) NOT = '-') 
                    THEN DO;
                        i_month  = INPUT(SUBSTR(O0400A5,5,2), 2.);    
                        i_day    = INPUT(SUBSTR(O0400A5,7,2), 2.);    
                        i_year   = INPUT(SUBSTR(O0400A5,1,4), 4.);    
                        d_O0400A5  = MDY(i_month, i_day, i_year);
                        *If d_O0400A5 is an invalid date then it will have a
                           missing value indicated in SAS by a period (.);
                        *Set earliest rehab date if d_O0400A5 is a valid date;
                        IF (d_O0400A5 = .)
                             THEN l_bad_date = 1;
                        ELSE d_rehab_start = d_O0400A5;
                    END;

               *If occupational therapy start date (O0400B5) present;
               IF (SUBSTR(O0400B5,1,1) NOT = '^' AND
                   SUBSTR(O0400B5,1,1) NOT = '-')
                    THEN DO;
                        i_month  = INPUT(SUBSTR(O0400B5,5,2), 2.);    
                        i_day    = INPUT(SUBSTR(O0400B5,7,2), 2.);    
                        i_year   = INPUT(SUBSTR(O0400B5,1,4), 4.);    
                        d_O0400B5  = MDY(i_month, i_day, i_year);
                        *If d_O0400B5 is an invalid date then it will have a
                          missing value indicated in SAS by a period (.);
                        *Set start of rehab date if d_O0400B5 is a valid date
                           that is earlier then the previous start of rehab 
                           date;
                        IF (d_O0400B5 = .)
                             THEN l_bad_date = 1;
                        ELSE DO;
                             IF (d_O0400B5 < d_rehab_start) 
                                 THEN d_rehab_start = d_O0400B5;
                        END;
                    END;

               *If physical therapy start date (O0400C5) present;
               IF (SUBSTR(O0400C5,1,1) NOT = '^' AND
                   SUBSTR(O0400C5,1,1) NOT = '-')
                    THEN DO;
                        i_month  = INPUT(SUBSTR(O0400C5,5,2), 2.);    
                        i_day    = INPUT(SUBSTR(O0400C5,7,2), 2.);    
                        i_year   = INPUT(SUBSTR(O0400C5,1,4), 4.);    
                        d_O0400C5  = MDY(i_month, i_day, i_year);
                        *If d_O0400C5 is an invalid date then it will have a
                          missing value indicated in SAS by a period (.);
                        *Set start of rehab date if d_O0400C5 is a valid date
                          that is earlier then the previous start of rehab 
                          date;
                        IF (d_O0400C5 = .)
                             THEN l_bad_date = 1;
                        ELSE DO; 
                             IF (d_O0400C5 < d_rehab_start) 
                                  THEN d_rehab_start = d_O0400C5;
                        END; 
                    END;

               *If rehab start date = initial value (1/1/2099), then no valid
                rehab start found and set bad date indicator (l_bad_date) = 1.;
               IF d_rehab_start = MDY(1, 1, 2099)
                    THEN l_bad_date = 1; 

          END;
     END;

/*
Handle end of therapy dates O0400A6, O0400B5, and O0400C6.
Determine latest end of rehab therapy date (d_rehab_end) and set bad 
date indicator (l_bad_date = 1) if no end of rehab therapy date. 

Set local variables for the end of therapy dates (e.g., s_O0400A6).  To
allow recoding of dates that indicate ongoing therapy (value  = '--------').
These ongoing therapy dates are set to the assessment reference date for
the following logic.
*/
s_O0400A6 = O0400A6;
IF (O0400A6 = '--------') THEN s_O0400A6 = A2300;
s_O0400B6 = O0400B6;
IF (O0400B6 = '--------') THEN s_O0400B6 = A2300;
s_O0400C6 = O0400C6;
IF (O0400C6 = '--------') THEN s_O0400C6 = A2300;

IF l_bad_date = 0
     THEN DO;
     
          *Initialize rehab therapy end date to very early date for 
           comparison;
          d_rehab_end = MDY(1, 1, 2000);

          *Check for no end of rehab therapy date;
          IF  ( (SUBSTR(s_O0400A6,1,1) = '^' OR SUBSTR(s_O0400A6,1,1) = '-') AND
                (SUBSTR(s_O0400B6,1,1) = '^' OR SUBSTR(s_O0400B6,1,1) = '-') AND
                (SUBSTR(s_O0400C6,1,1) = '^' OR SUBSTR(s_O0400C6,1,1) = '-') )
                     THEN l_bad_date = 1;
          ELSE DO;
               *If speech therapy end date (s_O0400A6) present;
               IF (SUBSTR(s_O0400A6,1,1) NOT = '^' AND
                   SUBSTR(s_O0400A6,1,1) NOT = '-') 
                    THEN DO;
                        i_month  = INPUT(SUBSTR(s_O0400A6,5,2), 2.);    
                        i_day    = INPUT(SUBSTR(s_O0400A6,7,2), 2.);    
                        i_year   = INPUT(SUBSTR(s_O0400A6,1,4), 4.);    
                        d_O0400A6  = MDY(i_month, i_day, i_year);
                        *If d_O0400A6 is an invalid date then it will have a
                           missing value indicated in SAS by a period (.);
                        *Set earliest rehab date if d_O0400A6 is a valid date;
                        IF (d_O0400A6 = .)
                             THEN l_bad_date = 1;
                        ELSE d_rehab_end = d_O0400A6;
                    END;

               *If occupational therapy end date (s_O0400B6) present;
               IF (SUBSTR(s_O0400B6,1,1) NOT = '^' AND
                   SUBSTR(s_O0400B6,1,1) NOT = '-')
                    THEN DO;
                        i_month  = INPUT(SUBSTR(s_O0400B6,5,2), 2.);    
                        i_day    = INPUT(SUBSTR(s_O0400B6,7,2), 2.);    
                        i_year   = INPUT(SUBSTR(s_O0400B6,1,4), 4.);    
                        d_O0400B6  = MDY(i_month, i_day, i_year);
                        *If d_O0400B6 is an invalid date then it will have a
                          missing value indicated in SAS by a period (.);
                        *Set end of rehab date if d_O0400B6 is a valid date
                           that is later then the previous end of rehab 
                           date;
                        IF (d_O0400B6 = .)
                             THEN l_bad_date = 1;
                        ELSE DO; 
                             IF (d_O0400B6 > d_rehab_end)
                                 THEN d_rehab_end = d_O0400B6;
                        END;
                    END;

               *If physical therapy end date (s_O0400C6) present;
               IF (SUBSTR(s_O0400C6,1,1) NOT = '^' AND
                    SUBSTR(s_O0400C6,1,1) NOT = '-')
                     THEN DO;
                         i_month  = INPUT(SUBSTR(s_O0400C6,5,2), 2.);    
                         i_day    = INPUT(SUBSTR(s_O0400C6,7,2), 2.);    
                         i_year   = INPUT(SUBSTR(s_O0400C6,1,4), 4.);    
                         d_O0400C6  = MDY(i_month, i_day, i_year);
                         *If d_O0400C6 is an invalid date then it will have a
                           missing value indicated in SAS by a period (.);
                         *Set end of rehab date if d_O0400C6 is a valid date
                           that is later then the previous end of rehab 
                           date;
                        IF (d_O0400C6 = .)
                             THEN l_bad_date = 1;
                        ELSE DO; 
                             IF (d_O0400C6 > d_rehab_end) 
                                  THEN d_rehab_end = d_O0400C6;
                        END;
                     END;

               *If rehab end = initial value (1/1/2000), then no valid
                rehab end found and set bad date indicator (l_bad_date) = 1.;
               IF d_rehab_end = MDY(1, 1, 2000)
                    THEN l_bad_date = 1; 

          END;
     END;

* Set Medicare Short Stay Indicator (l_Mcare_short_stay);
l_Mcare_short_stay = 0;
*V1.03.0 CHANGE to include new sRehabType = 'MCAR3';
IF ( (l_bad_date = 0) AND
     ((sRehabType = 'MCARE') OR (sRehabType = 'MCAR2') OR 
         (sRehabType = 'MCAR3')) AND
     (A0310C = '1') AND
     ((d_A2300 - d_A2400B) <= 7) AND
     (d_A2300 = d_A2400C) AND
     ((d_A2400C - d_rehab_start) <= 3) AND
     (d_A2400C = d_rehab_end) )
          THEN l_Mcare_short_stay = 1;

/*---------------------------------------------------------------------------
STEP #2
If the Medicare Short Stay Indicator = 1, then calculate the Medicare 
Short Stay Average Therapy Minutes. 
*/
n_avg_rehab_min = 0;
IF (l_Mcare_short_stay = 1)
     THEN n_avg_rehab_min = 
         (n_tot_rehab_min / (d_A2300 - d_rehab_start + 1));


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
SET DEPRESSION INDICATOR.

The depression indicator (l_depression) is set on the basis of the resident
interview PHQ summary score or the staff assessment PHQ summary score.
*/

l_depression = 0;
IF (INPUT(D0300, ??2.) >= '10' AND INPUT(D0300, ??2.) <= '27')
          THEN l_depression = 1; 

IF (INPUT(D0600, ??2.) >= '10' AND INPUT(D0600, ??2.) <= '30')
          THEN l_depression = 1; 

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
CALCULATE COUNT OF RESTORATIVE NURSING SERVICES RECEIVED FOR 6 OR MORE
DAYS.

Add one to the restorative nursing count (i_rnursing_cnt) for each of the 
following conditions that are satisfied.

1. Either urinary toileting program (H0200C) or bowel toileting 
   program (H0500) currently in use.
2. Either O0500A (passive ROM) or O0500B (active ROM) for 6 or more days.
3. O0500C (splint or brace assistance) for 6 or more days.
4. O0500D (bed mobility) or O0500F (walking training) for 6 or more days.
5. O0500E (transfer training) for 6 or more days.
6. O0500G (dressing or grooming training) for 6 or more days.
7. O0500H (eating or swallowing training) for 6 or more days.
8. O0500I (amputation/Prosthesis care) for 6 or more days.
9. O0500J (communication training) for 6 or more days.

*/

i_rnursing_cnt = 0;
IF (H0200C = '1' OR H0500 = '1')
          THEN i_rnursing_cnt = i_rnursing_cnt + 1;
IF ( (O0500A = '6' OR O0500A = '7') OR
     (O0500B = '6' OR O0500B = '7') )
          THEN i_rnursing_cnt = i_rnursing_cnt + 1;
IF (O0500C = '6' OR O0500C = '7') 
          THEN i_rnursing_cnt = i_rnursing_cnt + 1;
IF ( (O0500D = '6' OR O0500D = '7') OR
     (O0500F = '6' OR O0500F = '7') )
          THEN i_rnursing_cnt = i_rnursing_cnt + 1;
IF (O0500E = '6' OR O0500E = '7') 
          THEN i_rnursing_cnt = i_rnursing_cnt + 1;
IF (O0500G = '6' OR O0500G = '7') 
          THEN i_rnursing_cnt = i_rnursing_cnt + 1;
IF (O0500H = '6' OR O0500H = '7') 
          THEN i_rnursing_cnt = i_rnursing_cnt + 1;
IF (O0500I = '6' OR O0500I = '7') 
          THEN i_rnursing_cnt = i_rnursing_cnt + 1;
IF (O0500J = '6' OR O0500J = '7') 
          THEN i_rnursing_cnt = i_rnursing_cnt + 1;


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
SET RESTORATIVE NURSING INDICATOR.

The restorative nursing indicator (l_rnursing) is set if the restorative
nursing count (i_rnursing_cnt) is 2 or more.
*/

l_rnursing = 0;
IF (i_rnursing_cnt >= 2) 
          THEN l_rnursing = 1; 


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
DETERMINE LEVEL OF EXTENSIVE SERVICES.

The 3 extensive services are as follows:
   O0100E2  Tracheostomy care while a resident
   O0100F2  Ventilator or respirator while a resident
   O0100M2  Infection isolation while a resident

Level of extensive services (i_ext_serv_level) is determined as follows:
   Level = 3 if BOTH tracheostomy care AND ventilator/respirator
   Level = 2 if EITHER tracheostomy care OR ventilator/respirator
   Level = 1 if infection isolation WITHOUT tracheostomy care AND
             WITHOUT ventilator/respirator
   Level = 0 if NO tracheostomy care AND NO ventilator/respirator AND
             NO infection isolation
*/

i_ext_serv_level = 0;
IF ( (O0100E2 = '1') AND (O0100F2 = '1') )
     THEN i_ext_serv_level = 3;
ELSE IF ( (O0100E2 = '1') OR (O0100F2 = '1') )
     THEN i_ext_serv_level = 2;
ELSE IF (O0100M2 = '1')
     THEN i_ext_serv_level = 1;


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
DETERMINE EXTENSIVE SERVICES CATEGORY CLASSIFICATION.
 
The resident is classified in the Extensive Services Category
(l_ext_serv) if there is one or more of the extensive services are 
received (i_ext_serv_level >= 1) and the ADL score (i_adl_tot)
is 2 or more.
*/

l_ext_serv = 0;
IF ( (i_ext_serv_level >= 1) AND (i_adl_tot >= 2) )
     THEN l_ext_serv = 1;


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
DETERMINE REHABILITATION THERAPY SERVICES CATEGORY CLASSIFICATION.

There are 5 rehabilitation therapy services categories based on therapy
intensity:
     Ultra High Intensity
     Very High Intensity
     High Intensity
     Medium Intensity
     Low Intensity
*/

/*---------------------------------------------------------------------------
Ultra High Intensity Criteria

The resident qualifies for ultra high intensity if either (1) or (2) is satisfied.
     (1) In the last 7 days all 3 of the following conditions are satisfied:
               Total Therapy Minutes (calculated above) of 720 minutes or more
               (n_tot_rehab_min >= 720),
                    AND
               One discipline (O0400A4, O0400B4 or O0400C4) for at least 5 
               days (i_rehab_type_5days >= 1), 
                    AND
               A second discipline (O0400A4, O0400B4 or O0400C4) for at least 
               3 days (i_rehab_type_3days >= 2).

     (2) If this is a Medicare Short Stay Assessment (l_Mcare_short_stay = 1) 
         and the following condition is satisfied:
               Medicare Short Stay Average Therapy Minutes of 144 minutes or
               more (n_avg_rehab_min >= 144).  
*/

*Determine Ultra High Therapy indicator (l_rehab_ultra).;
l_rehab_ultra = 0;
IF ((n_tot_rehab_min >= 720) AND
    (i_rehab_type_5days >= 1) AND
    (i_rehab_type_3days >= 2) )
          THEN l_rehab_ultra = 1;

ELSE IF ( (l_Mcare_short_stay = 1) AND
          (n_avg_rehab_min >= 144) )
               THEN l_rehab_ultra = 1;
     
/*---------------------------------------------------------------------------
Very High Intensity Criteria

The resident qualifies for very high intensity if either (1) or (2) is satisfied.
     (1) In the last 7 days both of the following conditions are satisfied:
               Total Therapy Minutes (calculated above) of 500 minutes or more
               (n_tot_rehab_min >= 500),
                    AND
               One discipline (O0400A4, O0400B4 or O0400C4) for at least 5 
               days (i_rehab_type_5days >= 1). 

     (2) If this is a Medicare Short Stay Assessment (l_Mcare_short_stay = 1) 
         and the following condition is satisfied:
               Medicare Short Stay Average Therapy Minutes of 100 minutes or
               more (n_avg_rehab_min >= 100).  
*/

*Determine Very High Therapy indicator (l_rehab_very).;
l_rehab_very = 0;
IF ((n_tot_rehab_min >= 500) AND
    (i_rehab_type_5days >= 1) )
          THEN l_rehab_very = 1;

ELSE IF ( (l_Mcare_short_stay = 1) AND
          (n_avg_rehab_min >= 100) )
               THEN l_rehab_very = 1;

/*---------------------------------------------------------------------------
High Intensity Criteria

The resident qualifies for high intensity if either (1) or (2) is satisfied.
     (1) In the last 7 days both of the following conditions are satisfied:
               Total Therapy Minutes (calculated above) of 325 minutes or more
               (n_tot_rehab_min >= 325),
                    AND
               One discipline (O0400A4, O0400B4 or O0400C4) for at least 5 
               days (i_rehab_type_5days >= 1). 

     (2) If this is a Medicare Short Stay Assessment (l_Mcare_short_stay = 1) 
         and the following condition is satisfied:
               Medicare Short Stay Average Therapy Minutes of 65 minutes or
               more (n_avg_rehab_min >= 65).  
*/

*Determine High Therapy indicator (l_rehab_high).;
l_rehab_high = 0;
IF ((n_tot_rehab_min >= 325) AND
    (i_rehab_type_5days >= 1) )
          THEN l_rehab_high = 1;

ELSE IF ( (l_Mcare_short_stay = 1) AND
          (n_avg_rehab_min >= 65) )
               THEN l_rehab_high = 1;
           
/*---------------------------------------------------------------------------
Medium Intensity Criteria

The resident qualifies for medium intensity if (1), (2), or (3) is satisfied.
     (1) IF sRehabType = 'MCARE', 'MCAR2', or 'OTHER' and, in the last 7 days, 
         both of the following conditions are satisfied:
               Total Therapy Minutes (calculated above) of 150 minutes or more
               (n_tot_rehab_min >= 150),
                    AND
               The total days for the 3 disciplines (O0400A4, plus O0400B4 
               plus O0400C4) is 5 or more (i_tot_rehab_days >= 5).

     (2) IF sRehabType = 'MCAR3' and, in the last 7 days, both of the following 
         conditions are satisfied:
               Total Therapy Minutes (calculated above) of 150 minutes or more
               (n_tot_rehab_min >= 150),
                    AND
               The total distinct calendar days of therapy is 5 or more days 
               (O0420 >= 5).

                   NOTE: New condition added with Version 1.03.0.

     (3) If this is a Medicare Short Stay Assessment (l_Mcare_short_stay = 1) 
         and the following condition is satisfied:
               Medicare Short Stay Average Therapy Minutes of 30 minutes or
               more (n_avg_rehab_min >= 30).  
*/

*Determine Medium Therapy indicator (l_rehab_medium).;
l_rehab_medium = 0;
*V1.03.0 CHANGE to accomodate new sRehabType = 'MCAR3';
IF ( (sRehabType = 'MCARE' OR sRehabType ='MCAR2' OR sRehabType ='OTHER') AND
     (n_tot_rehab_min >= 150) AND
     (i_tot_rehab_days >= 5) )
          THEN l_rehab_medium = 1;

*V1.03.0 CHANGE to include new sRehabType = 'MCAR3';
ELSE IF ( (sRehabType = 'MCAR3') AND
          (n_tot_rehab_min >= 150) AND
          (O0420 >= 5) )
               THEN l_rehab_medium = 1;

ELSE IF ( (l_Mcare_short_stay = 1) AND
          (n_avg_rehab_min >= 30) )
               THEN l_rehab_medium = 1;

/*---------------------------------------------------------------------------
Low Intensity Criteria

The resident qualifies for low intensity if either (1), (2), or (3) is satisfied.
     (1) IF sRehabType = 'MCARE', 'MCAR2', or 'OTHER' and, in the last 7 days, 
         all 3 of the following conditions are satisfied:
               (n_tot_rehab_min >= 45),
                    AND
               The total days for the 3 disciplines (O0400A4, plus O0400B4 
               plus O0400C4) is 3 or more (i_tot_rehab_days >= 3),
                    AND
               2 or more restorative nursing services received for 6 or more
               days for at least 15 minutes a day (i_rnursing_cnt >= 2).

     (2) IF sRehabType = 'MCAR3' and, in the last 7 days, all 3 of the following 
         conditions are satisfied:
               (n_tot_rehab_min >= 45),
                    AND
               The total distinct calendar days of therapy is 3 or more 
               (O0420 >= 3),
                    AND
               2 or more restorative nursing services received for 6 or more
               days for at least 15 minutes a day (i_rnursing_cnt >= 2).

                   NOTE: New condition added with Version 1.03.0.

     (3) If this is a Medicare Short Stay Assessment (l_Mcare_short_stay = 1) 
         and the following condition is satisfied:
               Medicare Short Stay Average Therapy Minutes of 15 minutes or
               more (n_avg_rehab_min >= 15).  
*/

*Determine Low Therapy indicator (l_rehab_low).;
l_rehab_low = 0;
*V1.03.0 CHANGE to accomodate new sRehabType = 'MCAR3';
IF ( (sRehabType = 'MCARE' OR sRehabType ='MCAR2' OR sRehabType ='OTHER') AND
     (n_tot_rehab_min >= 45) AND
     (i_tot_rehab_days >= 3) AND
     (i_rnursing_cnt >= 2) )
          THEN l_rehab_low = 1;

ELSE IF ( (sRehabType = 'MCAR3') AND
          (n_tot_rehab_min >= 45) AND
          (O0420 >= 3) AND
          (i_rnursing_cnt >= 2) )
          THEN l_rehab_low = 1;

ELSE IF ( (l_Mcare_short_stay = 1) AND
          (n_avg_rehab_min >= 15) )
               THEN l_rehab_low = 1;

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
DETERMINE SPECIAL CARE HIGH CLASSIFICATION.
             
The resident has a Special Care High condition (l_sp_hi_cond) if any of the
following 7 conditions is present: 
   1. Comatose (B0100) and all self-performance late-loss ADLs
     (G0110A1, G0110B1, G0110H1, and G0110I1) are completely dependent 
      or activity did occur.
   2. Septicemia (I2100).
   3. Diabetes (I2900) with BOTH insulin injections (N0350A) for all 7 days 
      AND insulin order changes (N0350B) on 2 or more days.
   4. Quadriplegia (I5100) with ADL score (i_adl_tot) of 5 or more.
   5. Chronic obstructive pulmonary disease (I6200) and shortness of breath 
      (J1100C) when lying flat.
   6. Fever (J1550A) and one of the following:
         Pneumonia (I2000)
         Vomiting (J1550B)
         Any weight loss (K0300) 
         Feeding tube and minimum intake amount as indicated by one of the
            following 3 conditions:
              (a) For assessment reference date (A2300) before 4/1/2012:
                     Feeding tube (K0500B) with either 
                         51% or more total calories (K0700A)  
                            OR
                         26% to 50% of total calories (K0700A) and 501 cc or 
                               more per day (K0700B). 
              (b) For assessment reference date (A2300) on or after 4/1/2012 
                  and before 10/1/2013:
                     Feeding tube (K0510B1 or K0510B2) with either 
                         51% or more total calories (K0700A)  
                            OR
                         26% to 50% of total calories (K0700A) and 501 cc or 
                               more per day (K0700B).
                     NOTE: This condition changed with Version 1.03.0. 
             (c) For assessment reference date (A2300) on or after 10/1/2013:
                     Feeding tube (K0510B1 or K0510B2) with either 
                         51% or more total calories (K0710A3)  
                            OR
                         26% to 50% of total calories (K0710A3) and 501 cc  
                               or more per day (K0710B3). 
                     NOTE: This condition added with Version 1.03.0. 
   7. Parenteral/IV feedings: 
         (a) For assessment reference date (A2300) before 4/1/2012:
                 Parenteral/IV feedings (K0500A).
         (a) For assessment reference date (A2300) on or after 4/1/2012:
                 Parenteral/IV feedings (K0510A1 or K0510A2).
   8. Respiratory therapy (O0400D2) for all 7 days.

The resident is classified in the Special Care High Category (l_special_high) 
if there is one or more Special High conditions and the ADL score (i_adl_tot)
is 2 or more.  
*/

*Determine Special Care High condition indicator (l_sp_high_cond).;
l_sp_high_cond = 0;
*Condition 1;
IF ( (B0100 = '1') AND 
     (G0110A1 = '4' OR G0110A1 = '8') AND 
     (G0110B1 = '4' OR G0110B1 = '8') AND
     (G0110H1 = '4' OR G0110H1 = '8') AND
     (G0110I1 = '4' OR G0110I1 = '8') ) 
          THEN l_sp_high_cond = 1;
*Condition 2;
IF (I2100 = '1')
     THEN l_sp_high_cond = 1;
*Condition 3;
IF ( (I2900 = '1') AND 
     (N0350A = '7') AND 
     (N0350B >= '2') )
          THEN l_sp_high_cond = 1;
*Condition 4;
IF ( (I5100 = '1') AND 
     (i_adl_tot >= 5) )
          THEN l_sp_high_cond = 1;
*Condition 5;
IF ( (I6200 = '1') AND 
     (J1100C = '1') )
          THEN l_sp_high_cond = 1;
*Condition 6;
*V1.03.0 CHANGE to use new K0710 items instead of K0700 items on or 
 after 10/1/2013;
IF ( (J1550A = '1') AND 
        ( (I2000 = '1') OR
          (J1550B = '1') OR
          (K0300 = '1' OR K0300 = '2') OR
          ( (A2300 < '20120401') AND
            ( (K0500B = '1' AND K0700A = '3') OR
              (K0500B = '1' AND K0700A = '2' AND K0700B = '2') ) ) OR
          ( (A2300 >= '20120401' AND A2300 < '20131001') AND
            ( ( (K0510B1 = '1' OR K0510B2 = '1') AND K0700A = '3') OR
                ((K0510B1 = '1' OR K0510B2 = '1') AND 
                 (K0700A = '2' AND K0700B = '2') ) ) ) OR 
          ( (A2300 >= '20131001') AND
            ( ( (K0510B1 = '1' OR K0510B2 = '1') AND K0710A3 = '3') OR
                ( (K0510B1 = '1' OR K0510B2 = '1') AND 
                  (K0710A3 = '2' AND K0710B3 = '2') ) ) ) ) )
                   THEN l_sp_high_cond = 1;
*Condition 7;
IF (A2300 < '20120401') AND (K0500A = '1')
     THEN l_sp_high_cond = 1;
IF (A2300 >= '20120401') AND (K0510A1 = '1' OR K0510A2 = '1')
     THEN l_sp_high_cond = 1;
*Condition 8;
IF (O0400D2 = '7')
     THEN l_sp_high_cond = 1;

*Check ADL score and determine Special Care High category classification
   indicator (l_special_high).;
l_special_high = 0;
IF ( (l_sp_high_cond = 1) AND (i_adl_tot >= 2) )
     THEN l_special_high = 1;


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
DETERMINE SPECIAL CARE LOW CLASSIFICATION.
             
The resident as a Special Care Low condition if any of the following 12 
conditions is satisfied: 
   1. Cerebral palsy (I4400) with ADL score (i_adl_tot) of 5 or more.
   2. Multiple sclerosis (I5200), with ADL score of 5 or more.
   3. Parkinsons disease (I5300) with ADL score of 5 or more.
   4. Respiratory failure (I6300) and oxygen therapy (O0100C2) while a 
      resident.
   5. Feeding tube with minimum intake amount:
           (a) For assessment reference date (A2300) on or after 4/1/2012
               and before 10/1/2013:
                  Feeding tube (K0500B) with either 
                     51% or more total calories (K0700A)  
                         OR
                      26% to 50% of total calories (K0700A) and 501 cc or 
                            more per day (K0700B). 
           (b) For assessment reference date (A2300) on or after 4/1/2012 
               and before 10/1/2013:
                  Feeding tube (K0510B1 or K0510B2) with either 
                      51% or more total calories (K0700A)  
                         OR
                      26% to 50% of total calories (K0700A) and 501 cc or 
                            more per day (K0700B). 
                     NOTE: This condition changed with Version 1.03.0. 
           (c) For assessment reference date (A2300) on or after 10/1/2012 
                  and before 10/1/2014:
                     Feeding tube (K0510B1 or K0510B2) with either 
                         51% or more total calories (K0710A3)  
                            OR
                         26% to 50% of total calories (K0710A3) and 501 cc  
                               or more per day (K0710B3). 
                     NOTE: This condition added with Version 1.03.0. 
   6. 2 or more Stage 2 pressure ulcers (M0300B1) with 2 or more Special
      Low selected skin treatments (see below).
   7. Any Stage 3 or 4 pressure ulcer (M0300C1, M0300D1, and M0300F1
      with 2 or more Special Low selected skin treatments (see below).
   8. 2 or more venous/arterial ulcers (M1030) with 2 or more Special Low
      selected skin treatments (see below).
   9. 1 Stage 2 pressure ulcer (M0300B1) and 1 venous/arterial ulcer
      (M1030) with 2 or more Special Low selected skin treatments (see
      below).
  10. Foot infection (M1040A), diabetic foot ulcer (M1040B) or other open 
      lesion of foot (M1040C) with application of dressings to the feet
      (M1200I).
  11. Radiation treatment while a resident (O0100B2).
  12. Dialysis treatment while a resident (O0100J2).

The resident is classified in the Special Care Low Category (l_special_low) 
if there is one or more Special Low conditions and the ADL score (i_adl_tot)
is 2 or more.  

Special Low Category Selected Skin Treatment Count (i_SL_skin_tx_cnt).
     Add 1 to this count for each of the following conditions that is
     satisfied:
          a. Pressure relieving chair (M1200A) and/or bed (M1200B).
          b. Turning/repositioning (M1200C).
          c. Nutrition or hydration intervention (M1200D).
          d. Ulcer care (M1200E).
          e. Application of dressings (not to feet) (M1200G).
          f. Application of ointments (not to feet) (M1200H).
*/

*Calculate the Special Low category selected skin treatment count
   (i_sl_skin_tx_cnt);
i_sl_skin_tx_cnt = 0;
IF ( (M1200A = '1' OR M1200B = '1') ) 
     THEN i_sl_skin_tx_cnt = i_sl_skin_tx_cnt +1;
IF (M1200C = '1') 
     THEN i_sl_skin_tx_cnt = i_sl_skin_tx_cnt +1;
IF (M1200D = '1') 
     THEN i_sl_skin_tx_cnt = i_sl_skin_tx_cnt +1;
IF (M1200E = '1') 
     THEN i_sl_skin_tx_cnt = i_sl_skin_tx_cnt +1;
IF (M1200G = '1') 
     THEN i_sl_skin_tx_cnt = i_sl_skin_tx_cnt +1;
IF (M1200H = '1') 
     THEN i_sl_skin_tx_cnt = i_sl_skin_tx_cnt +1;

*Determine Special Care Low condition indicator (l_sp_low_cond).;
l_sp_low_cond = 0;
*Condition 1;
IF ( (I4400 = '1') AND 
     (i_adl_tot >= 5) )
          THEN l_sp_low_cond = 1;
*Condition 2;
IF ( (I5200 = '1') AND 
     (i_adl_tot >= 5) )
          THEN l_sp_low_cond = 1;
*Condition 3;
IF ( (I5300 = '1') AND 
     (i_adl_tot >= 5) )
          THEN l_sp_low_cond = 1;
*Condition 4;
IF ( (I6300 = '1') AND 
     (O0100C2 = '1') )
          THEN l_sp_low_cond = 1;
*Condition 5;
*V1.03.0 CHANGE to use new K0710 items instead of K0700 items on or 
 after 10/1/2013;
IF ( ( (A2300 < '20120401') AND
       ( (K0500B = '1' AND K0700A = '3') OR
         (K0500B = '1' AND K0700A = '2' AND K0700B = '2') ) ) OR
     ( (A2300 >= '20120401' AND A2300 < '20131001') AND
       ( ( (K0510B1 = '1' OR K0510B2 = '1') AND K0700A = '3') OR
         ( (K0510B1 = '1' OR K0510B2 = '1') AND 
           (K0700A = '2' AND K0700B = '2') ) ) ) OR
     ( (A2300 >= '20131001') AND
       ( ( (K0510B1 = '1' OR K0510B2 = '1') AND K0710A3 = '3') OR
         ( (K0510B1 = '1' OR K0510B2 = '1') AND 
           (K0710A3 = '2' AND K0710B3 = '2') ) ) ) )
                THEN l_sp_low_cond = 1;
*Condition 6;
IF ( (M0300B1 >= '2' AND M0300B1 <= '9') AND
     (i_sl_skin_tx_cnt >= 2) )
          THEN l_sp_low_cond = 1;
*Condition 7;
IF ( ( (M0300C1 >= '1' AND M0300C1 <= '9') OR
       (M0300D1 >= '1' AND M0300D1 <= '9') OR
       (M0300F1 >= '1' AND M0300F1 <= '9') )
            AND
       (i_sl_skin_tx_cnt >= 2) )
            THEN l_sp_low_cond = 1;
*Condition 8;
IF ( (M1030 >= '2' AND M1030 <= '9') AND
     (i_sl_skin_tx_cnt >= 2) )
          THEN l_sp_low_cond = 1;
*Condition 9;
IF ( (M0300B1 = '1' AND M1030 = '1') AND
     (i_sl_skin_tx_cnt >= 2) )
          THEN l_sp_low_cond = 1;
*Condition 10;
IF ( ( (M1040A = '1') OR
       (M1040B = '1') OR
       (M1040C = '1') )
            AND
       (M1200I = '1') )
          THEN l_sp_low_cond = 1;
*Condition 11;
IF (O0100B2 = '1')
          THEN l_sp_low_cond = 1;
*Condition 12;
IF (O0100J2 = '1')
          THEN l_sp_low_cond = 1;

*Check ADL score and determine Special Care Low category classification
indicator (l_special_low).;
l_special_low = 0;
IF ( (l_sp_low_cond = 1) AND (i_adl_tot >= 2) )
     THEN l_special_low = 1;


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
DETERMINE CLINICALLY COMPLEX CLASSIFICATION.
             
The resident qualifies for Clinically Complex if any of the following 11 
conditions is satisfied: 

   1. Pneumonia (I2000).
   2. Hemiplegia/hemiparesis (I4900) with ADL score (i_adl_tot) of 5 or more.
   3. Surgical wounds (M1040E) or open lesions (M1040D) with any Clinically
      Complex selected skin treatment (see below).
   4. Burns (M1040F).
   5. Chemotherapy while a resident (O0100A2).
   6. Oxygen therapy while a resident (O0100C2).
   7. IV medications while a resident (O0100H2).
   8. Transfusions while a resident (O0100I2).
   9. Resident receives extensive services (i_ext_serv_level >= 1) with 
      ADL score (i_adl_tot) of 0 or 1.
  10. Resident has a special care high condition (l_sp_high_cond >= 1) with 
      ADL score (i_adl_tot) of 0 or 1.
  11. Resident has a special care low condition (i_sp_low_cond >= 1) with 
      ADL score (i_adl_tot) of 0 or 1.

Clinically Complex Category Selected Skin Treatment Count (i_cc_skin_tx_cnt).
     Add 1 to this count for each of the following conditions that is
     satisfied:
          a. Surgical wound care (M1200F).
          b. Application of dressing (not to feet) (M1200G).
          c. Application of ointments (not to feet) (M1200H).
*/

*Calculate the Clinically Complex category selected skin treatment count
   (i_cc_skin_tx_cnt);
i_cc_skin_tx_cnt = 0;
IF (M1200F = '1') 
     THEN i_cc_skin_tx_cnt = i_cc_skin_tx_cnt +1;
IF (M1200G = '1') 
     THEN i_cc_skin_tx_cnt = i_cc_skin_tx_cnt +1;
IF (M1200H = '1') 
     THEN i_cc_skin_tx_cnt = i_cc_skin_tx_cnt +1;

*Determine Clinically Complex category classification indicator 
   (l_clin_complex).;
l_clin_complex = 0;
*Condition 1;
IF (I2000 = '1') 
     THEN l_clin_complex = 1;
*Condition 2;
IF ( (I4900 = '1') AND
     (i_adl_tot >= 5) )
          THEN l_clin_complex = 1;
*Condition 3;
IF ( ( (M1040E = '1') OR (M1040D = '1') ) AND
     (i_cc_skin_tx_cnt >= 1) )
          THEN l_clin_complex = 1;
*Condition 4;
IF (M1040F = '1') 
     THEN l_clin_complex = 1;
*Condition 5;
IF (O0100A2 = '1') 
     THEN l_clin_complex = 1;
*Condition 6;
IF (O0100C2 = '1') 
     THEN l_clin_complex = 1;
*Condition 7;
IF (O0100H2 = '1') 
     THEN l_clin_complex = 1;
*Condition 8;
IF (O0100I2 = '1') 
     THEN l_clin_complex = 1;
*Condition 9;
IF ( (i_ext_serv_level >= 1) AND
     (i_adl_tot <= 1) )
          THEN l_clin_complex = 1;
*Condition 10;
IF ( (l_sp_high_cond = 1) AND
     (i_adl_tot <= 1) )
          THEN l_clin_complex = 1;
*Condition 11;
IF ( (l_sp_low_cond = 1) AND
     (i_adl_tot <= 1) )
          THEN l_clin_complex = 1;


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
DETERMINE BEHAVIORAL SYMPTOMS AND COGNITIVE PERFORMANCE CLASSIFICATION.

This category includes residents with cognitive impairment or behavioral
symptoms and an ADL score of 5 or less.

The resident qualifies for the Behavior Symptoms and Cognitive Performance
Classification according to the following logic:

   A. If the BIMS Brief Interview for Mental Status Summary Score (C0500) is 
      9 or less, the resident is cognitive impaired. The BIMS score is in item
      C0500.

      Note that the BIMS summary score was not recalculated from the component
      items (C0200 through C0400C).  If the interview was successfully 
      completed with the resident attempting to answer each item, then the
      summary score in C0500 will equal the sum of the component items. 
      However if the resident does not respond to any component item, this
      item will be coded as 0 but the assessor will code the summary score as
      99 indicating the interview was not successfully completed.   

   B. If the BIMS summary score is greater than 9 and not 99, the resident is
      not cognitively impaired.

   C. If the BIMS was not conducted (C0100 = 0) or the BIMS score is 99
      (indicating that all interview items could not be completed) or the
      BIMS score is - (not assessed, then the RUG-IV Cognitive Performance
      Scale (CPS) based on staff interview is used to assess cognitive
      impairment.  The CPS Scale indicates cognitive impairment if any of the
      following 3 conditions are satisfied:
        (1) The resident is comatose (B0100=1) and completely ADL
            dependent or ADL did not occur for all 4 late-loss ADLS
            (G0110A1, G0110B1, G0110H1, G010I1 all = 4 or 8). 
        (2) The resident has severely impaired cognitive skills
            (C1000 = 3).
        (3) Two or more of the following impairment indicators are present
            a. Problem being understood (B0700 > 0).
            b. Short-term memory problem (C0700 = 1).
            c. Cognitive skills problem (C1000 > 0).
                 AND
            One or more of the following severe impairment indicators are
            present:
            a. Severe problem being understood (B0700 >= 2).
            b. Severe cognitive skills problem (C1000 >= 2). 
           		
   D. The resident has behavior symptoms if any of the following 7 conditions
      are satisfied:
         (1) The resident has hallucinations (E0100A = 1).
         (2) The resident has delusions (E0100B = 1).
         (3) The resident displays physical behavioral symptoms directed 
             toward others on 4 or more of 7 days (E0200A = 2 or 3). 
         (4) The resident displays verbal behavioral symptoms directed 
             toward others on 4 or more of 7 days (E0200B = 2 or 3).
         (5) The resident displays other behavioral symptoms not directed
             toward others on 4 or more of 7 days (E0200C = 2 or 3).
         (6) The resident displays rejection of care on 4 or more of 7
             days (E0800 = 2 or 3).
         (7) The resident displays wandering on 4 or more of 7 days
             (E0900 = 2 or 3).
*/

*Initialize cognitive impairment indicator (l_cog_impair);
l_cog_impair = 0;

*Set cognitive impairment indicator (l_cog_impair) based on the BIMS.;
IF (INPUT(C0500, ??2.) >= 0 AND INPUT(C0500, ??2.) <= 09) 
     THEN l_cog_impair = 1;

/*IF BIMS not conducted indicated by BIMS score = ^ (skipped) or BIMS score
= 99, use the CPS Scale to determine cognitive impairment.
*/
IF ((SUBSTR(C0500,1,1) = '^') OR (C0500 = '99') OR (C0500 = '-'))
THEN DO;

     *Determine cognitive impairment based on the CPS scale; 
     IF ( (B0100 = '1') AND 
          ((G0110A1 = '4') OR (G0110A1 = '8')) AND
          ((G0110B1 = '4') OR (G0110B1 = '8')) AND
          ((G0110H1 = '4') OR (G0110H1 = '8')) AND
          ((G0110I1 = '4') OR (G0110I1 = '8')) )
               THEN l_cog_impair = 1;
     ELSE IF (C1000 = '3')
               THEN l_cog_impair = 1;
     ELSE DO;
          *Count number of cognitive impairment symptoms (i_cog_sympt);
          i_cog_sympt = 0;
          IF ( (B0700 > '0') AND (B0700 <= '3') )
               THEN i_cog_sympt = i_cog_sympt + 1;
          IF (C0700 = '1') 
               THEN i_cog_sympt = i_cog_sympt + 1;
          IF ( (C1000 > '0') AND (C1000 <= '3') )
               THEN i_cog_sympt = i_cog_sympt + 1;

          *Count number of severe cognitive impairment symptoms
            (i_sev_cog_sympt);
          i_sev_cog_sympt = 0;
          IF ( (B0700 >= '2') AND (B0700 <= '3') ) 
               THEN i_sev_cog_sympt = i_sev_cog_sympt + 1;
          IF ( (C1000 >= '2') AND (C1000 <= '3') )
               THEN i_sev_cog_sympt = i_sev_cog_sympt + 1;

          *Determine cognitive impairment based on these symptoms.;
          IF ( (i_cog_sympt >= 2) AND (i_sev_cog_sympt >= 1) )
               THEN l_cog_impair = 1;
     END;

END;

*Set behavioral symptoms indicator (l_beh_sympt);
l_beh_sympt = 0;
IF (E0100A = '1')
     THEN l_beh_sympt = 1;
IF (E0100B = '1')
     THEN l_beh_sympt = 1;
IF ( (E0200A >= '2') AND (E0200A <= '3') )
     THEN l_beh_sympt = 1;
IF ( (E0200B >= '2') AND (E0200B <= '3') )
     THEN l_beh_sympt = 1;
IF ( (E0200C >= '2') AND (E0200C <= '3') )
     THEN l_beh_sympt = 1;
IF ( (E0800 >= '2') AND (E0800 <= '3') )
     THEN l_beh_sympt = 1;
IF ( (E0900 >= '2') AND (E0900 <= '3') )
     THEN l_beh_sympt = 1;

/*Set Behavioral Symptoms and Cognitive Performance category indicator
   (l_beh_cog) if ADL score is <= 5 and either cognitive impairment or
   behavioral symptoms are present
*/
l_beh_cog = 0;
IF ( (i_adl_tot <= 5) AND
     ( (l_cog_impair = 1) OR (l_beh_sympt = 1) ) )
          THEN l_beh_cog = 1; 


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
Classify into RUG-IV Groups

Four different classification results are determined:
   1. sRugHier = RUG group resulting from hierarchical classification.
   2. sRugHier_NT = non-therapy (NT) RUG group resulting from hierarchical
         classification.  This classification excludes groups in the
         Rehabilitation Plus Extensive Services Category and the
         Rehabilitation Category. 
   3. sRugMax = RUG group resulting from index maximizing classification
         using the Case Mix Indices in the CMI array (nCmiArray).
   2. sRugMax_NT = non-therapy (NT) RUG group resulting from Index
         Maximizing classification.  This classification excludes groups
         in the Rehabilitation Plus Extensive Services Category and the
         Rehabilitation Category. 

NOTE:  nCmiValueMax is used to control both index maximizing and
hierarchical classification for normal classification (includes
Rehabilitation Plus Extensive Services groups and Rehabilitation groups).
nCmiValueMax is initialized to a value of -9999, and this initial value
indicates that neither index maximizing nor hierarchical normal
classification has been made yet. 

NOTE:  nCmiValueMax_NT is used to control both index maximizing and
hierarchical classification for non-therapy classification (excludes
Rehabilitation Plus Extensive Services groups and Rehabilitation groups).
nCmiValueMax_NT is initialized to a value of -9999, and this initial value
indicates that neither index maximizing nor hierarchical non-therapy
classification has been made yet. 
*/

*Initialize nCmiValueMax;
nCmiValueMax = -9999;

*Initialize nCmiValueMax_NT;
nCmiValueMax_NT = -9999;

/*===============================
Begin Classification into 66-group Rehab Plus Extensive Groups
*/
IF sModel = '66' 
THEN DO;

/*-------------------------------
Classify into Ultra High Intensity Rehabilitation Plus Extensive Services
groups.

   Classify if Ultra High Rehab indicator (l_rehab_ultra) and Extensive
   Services indicator (l_ext_serv) are both present. 

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax. 
*/
IF ( (l_rehab_ultra = 1) AND (l_ext_serv = 1) ) 
THEN DO;

   /*Determine final splits for Ultra High Rehab Plus Extensive residents on
     the basis of ADL score (i_adl_tot).
   */
   IF     (11 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF nCmiArray(1) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO; 
                   nCmiValueHier = nCmiArray(1);
                   sRugHier = 'RUX'; 
                   nRugHier = 1;
                END;
                nCmiValueMax = nCmiArray(1);
                sRugMax = 'RUX';
                nRugMax = 1;
             END;
   END;

   ELSE IF     (2 <= i_adl_tot  AND  i_adl_tot <= 10)
   THEN DO;
             IF nCmiArray(2) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999 
                THEN DO; 
                   nCmiValueHier = nCmiArray(2);
                   sRugHier = 'RUL'; 
                   nRugHier = 2;
                END;
                nCmiValueMax = nCmiArray(2);
                sRugMax = 'RUL';
                nRugMax = 2;
             END;
   END;
END;

/*-------------------------------
Classify into Very High Intensity Rehabilitation Plus Extensive Services
groups.

   Classify if Very High Rehab indicator (l_rehab_very) and Extensive
   Services indicator (l_ext_serv) are both present. 

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax. 
*/
IF ( (l_rehab_very = 1) AND (l_ext_serv = 1) ) 
THEN DO;

   /*Determine final splits for Very High Rehab Plus Extensive residents on
     the basis of ADL score (i_adl_tot).
   */
   IF     (11 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF nCmiArray(3) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999 
                THEN DO; 
                   nCmiValueHier = nCmiArray(3);
                   sRugHier = 'RVX'; 
                   nRugHier = 3;
                END;
                nCmiValueMax = nCmiArray(3);
                sRugMax = 'RVX';
                nRugMax = 3;
             END;
   END;

   ELSE IF     (2 <= i_adl_tot  AND  i_adl_tot <= 10)
   THEN DO;
             IF nCmiArray(4) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999 
                THEN DO; 
                   nCmiValueHier = nCmiArray(4);
                   sRugHier = 'RVL'; 
                   nRugHier = 4;
                END;
                nCmiValueMax = nCmiArray(4);
                sRugMax = 'RVL';
                nRugMax = 4;
             END;
   END;
END;

/*-------------------------------
Classify into High Intensity Rehabilitation Plus Extensive Services
groups.
   Classify if High Rehab indicator (l_rehab_high) and Extensive
   Services indicator (l_ext_serv) are both present. 

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax. 
*/
	
IF ( (l_rehab_high = 1) AND (l_ext_serv = 1) ) 
THEN DO;

   /*Determine final splits for High Rehab Plus Extensive residents on
     the basis of ADL score (i_adl_tot).
   */
   IF     (11 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF nCmiArray(5) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999 
                THEN DO; 
                   nCmiValueHier = nCmiArray(5);
                   sRugHier = 'RHX'; 
                   nRugHier = 5;
                END;
                nCmiValueMax = nCmiArray(5);
                sRugMax = 'RHX';
                nRugMax = 5;
             END;
   END;

   ELSE IF     (2 <= i_adl_tot  AND  i_adl_tot <= 10)
   THEN DO;
             IF nCmiArray(6) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999 
                THEN DO; 
                   nCmiValueHier = nCmiArray(6);
                   sRugHier = 'RHL'; 
                   nRugHier = 6;
                END;
                nCmiValueMax = nCmiArray(6);
                sRugMax = 'RHL';
                nRugMax = 6;
             END;
   END;
END;

/*-------------------------------
Classify into Medium Intensity Rehabilitation Plus Extensive Services
groups.
   Classify if Medium Rehab indicator (l_rehab_medium) and Extensive
   Services indicator (l_ext_serv) are both present. 

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax. 
*/
	
IF ( (l_rehab_medium = 1) AND (l_ext_serv = 1) ) 
THEN DO;

   IF     (11 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF nCmiArray(7) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999 
                THEN DO; 
                   nCmiValueHier = nCmiArray(7);
                   sRugHier = 'RMX'; 
                   nRugHier = 7;
                END;
                nCmiValueMax = nCmiArray(7);
                sRugMax = 'RMX';
                nRugMax = 7;
             END;
   END;

   ELSE IF     (2 <= i_adl_tot  AND  i_adl_tot <= 10)
   THEN DO;
             IF nCmiArray(8) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999 
                THEN DO; 
                   nCmiValueHier = nCmiArray(8);
                   sRugHier = 'RML'; 
                   nRugHier = 8;
                END;
                nCmiValueMax = nCmiArray(8);
                sRugMax = 'RML';
                nRugMax = 8;
             END;
   END;
END;

/*-------------------------------
Classify into the Low Intensity Rehabilitation Plus Extensive Services
group.
   Classify if Low Rehab indicator (l_rehab_low) and Extensive
   Services indicator (l_ext_serv) are both present. 

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax. 
*/
	
IF ( (l_rehab_low = 1) AND (l_ext_serv = 1) ) 
THEN DO;

   IF     (2 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF nCmiArray(9) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999 
                THEN DO; 
                   nCmiValueHier = nCmiArray(9);
                   sRugHier = 'RLX'; 
                   nRugHier = 9;
                END;
                nCmiValueMax = nCmiArray(9);
                sRugMax = 'RLX';
                nRugMax = 9;
             END;
   END;
END;


*End of classification into 66-group Rehab Plus Extensive Groups;
END;
*===============================;

/*===============================
Begin Classification into 66-group and 57-group Rehab Groups
*/
IF ( (sModel = '66') OR (sModel = '57') ) 
THEN DO;

/*-------------------------------
Classify into Ultra High Intensity Rehabilitation groups.

   Classify if Ultra High Rehab indicator (l_rehab_ultra) is present. 

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax. 
*/
IF (l_rehab_ultra = 1) 
THEN DO;

   IF     (11 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF nCmiArray(10) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO; 
                   nCmiValueHier = nCmiArray(10);
                   sRugHier = 'RUC'; 
                   nRugHier = 10; 
                END;
                nCmiValueMax = nCmiArray(10);
                sRugMax= 'RUC';
                nRugMax=10;
             END;
   END;

   ELSE IF ( 6 <= i_adl_tot  AND  i_adl_tot <= 10)
   THEN DO;
             IF nCmiArray(11) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO; 
                   nCmiValueHier = nCmiArray(11);
                   sRugHier = 'RUB';
                   nRugHier = 11;
                END;
                nCmiValueMax = nCmiArray(11);
                sRugMax = 'RUB';
                nRugMax = 11;
              END;
   END;

   ELSE IF ( 0 <= i_adl_tot  AND  i_adl_tot <=  5)  
   THEN DO;
             IF nCmiArray(12) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(12);
                   sRugHier = 'RUA';
                   nRugHier = 12;
                END;
                nCmiValueMax = nCmiArray(12);
                sRugMax = 'RUA';
                nRugMax = 12;
             END;
   END;
END;

/*-------------------------------
Classify into Very High Intensity Rehabilitation groups.

   Classify if Very High Rehab indicator (l_rehab_very) is present. 

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax. 
*/
IF (l_rehab_very = 1) 
THEN DO;

   IF     (11 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF nCmiArray(13) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(13);
                   sRugHier = 'RVC';
                   nRugHier = 13;
                END;
                nCmiValueMax = nCmiArray(13);
                sRugMax = 'RVC';
                nRugMax = 13;
             END;
   END;

   ELSE IF ( 6 <= i_adl_tot  AND  i_adl_tot <= 10)
   THEN DO;
             IF nCmiArray(14) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(14);
                   sRugHier = 'RVB';
                   nRugHier = 14;
                END;
                nCmiValueMax = nCmiArray(14);
                sRugMax = 'RVB';
                nRugMax = 14;
             END;
   END;

   ELSE IF ( 0 <= i_adl_tot  AND  i_adl_tot <=  5)
   THEN DO;
             IF nCmiArray(15) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(15);
                   sRugHier = 'RVA';
                   nRugHier = 15;
                END;
                nCmiValueMax = nCmiArray(15);
                sRugMax = 'RVA';
                nRugMax = 15;
             END;
   END;
END;

/*-------------------------------
Classify into High Intensity Rehabilitation groups.

   Classify if High Rehab indicator (l_rehab_high) is present. 

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax. 
*/
IF (l_rehab_high = 1) 
THEN DO;

   IF     (11 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF nCmiArray(16) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(16);
                   sRugHier = 'RHC';
                   nRugHier = 16;
                END;
                nCmiValueMax = nCmiArray(16);
                sRugMax = 'RHC';
                nRugMax = 16;
             END;
   END;

   ELSE IF ( 6 <= i_adl_tot  AND  i_adl_tot <= 10)
   THEN DO;
             IF nCmiArray(17) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(17);
                   sRugHier = 'RHB';
                   nRugHier = 17;
                END;
                nCmiValueMax = nCmiArray(17);
                sRugMax = 'RHB';
                nRugMax = 17;
             END;
   END;

   ELSE IF ( 0 <= i_adl_tot  AND  i_adl_tot <=  5)
   THEN DO;
             IF nCmiArray(18) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(18);
                   sRugHier = 'RHA';
                   nRugHier = 18;
                END;
                nCmiValueMax = nCmiArray(18);
                sRugMax = 'RHA';
                nRugMax = 18;
             END;
   END;
END;

/*-------------------------------
Classify into Medium Intensity Rehabilitation groups.

   Classify if Medium Rehab indicator (l_rehab_medium) is present. 

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax. 
*/
IF (l_rehab_medium = 1) 
THEN DO;

   IF     (11 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF nCmiArray(19) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(19);
                   sRugHier = 'RMC';
                   nRugHier = 19;
                END;
                nCmiValueMax = nCmiArray(19);
                sRugMax = 'RMC';
                nRugMax = 19;
             END;
   END;

   ELSE IF ( 6 <= i_adl_tot  AND  i_adl_tot <= 10)
   THEN DO;
             IF nCmiArray(20) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(20);
                   sRugHier = 'RMB';
                   nRugHier = 20;
                END;
                nCmiValueMax = nCmiArray(20);
                sRugMax = 'RMB';
                nRugMax = 20;
             END;
   END;

   ELSE IF ( 0 <= i_adl_tot  AND  i_adl_tot <=  5)
   THEN DO;
             IF nCmiArray(21) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(21);
                   sRugHier = 'RMA';
                   nRugHier = 21;
                END;
                nCmiValueMax = nCmiArray(21);
                sRugMax = 'RMA';
                nRugMax = 21;
             END;
   END;
END;

/*-------------------------------
Classify into Low Intensity Rehabilitation groups.

   Classify if low Rehab indicator (l_rehab_low) is present. 

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax. 
*/
IF (l_rehab_low = 1) 
THEN DO;

   IF     (11 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF nCmiArray(22) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(22);
                   sRugHier = 'RLB';
                   nRugHier = 22;
                END;
                nCmiValueMax = nCmiArray(22) ;
                sRugMax = 'RLB';
                nRugMax = 22;
             END;
   END;

   ELSE IF ( 0 <= i_adl_tot  AND  i_adl_tot <= 10)
   THEN DO;
             IF nCmiArray(23) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(23);
                   sRugHier = 'RLA';
                   nRugHier = 23;
                END;
                nCmiValueMax = nCmiArray(23);
                sRugMax = 'RLA';
                nRugMax = 23;
             END;
   END;

END;

*End of Classification into 66-group and 57-group Rehabilitation Groups;
 END;
*===============================;


/*===============================
Begin Classification into Extensive Services Groups for all models (66-,
57-, and 48-groups)

   Classify if Extensive Services category indicator (l_ext_serv) is present.
   This indicator is present if both Extensive Services are received and
   the ADL score is 2 or more.

   Note that residents who receive Extensive Services but have too low an
   ADL score (0 or 1) are classified as Clinically Complex rather than
   Extensive Services. 

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax.

   Both normal and non-therapy classifications are made for the Extensive
   Services groups. 
*/

IF  (l_ext_serv = 1)
THEN DO;

   /*Split into Extensive Services groups is based on the level of extensive
     Services (i_ext_serv_level). 
   */
   IF     (i_ext_serv_level = 3)
   THEN DO;
             *Handle normal classification;
             IF nCmiArray(24) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(24);
                   sRugHier = 'ES3';
                   nRugHier = 24;
                END;
                nCmiValueMax = nCmiArray(24) ;
                sRugMax = 'ES3';
                nRugMax = 24;
             END;

             *Handle non-therapy classification;
             IF nCmiArray(24) > nCmiValueMax_NT
             THEN DO;
                IF nCmiValueMax_NT = -9999
                THEN DO;
                   nCmiValueHier_NT = nCmiArray(24);
                   sRugHier_NT = 'ES3';
                   nRugHier_NT = 24;
                END;
                nCmiValueMax_NT = nCmiArray(24) ;
                sRugMax_NT = 'ES3';
                nRugMax_NT =24;
             END;

   END;

   ELSE IF     (i_ext_serv_level = 2)
   THEN DO;
             *Handle normal classification;
             IF nCmiArray(25) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(25);
                   sRugHier = 'ES2';
                   nRugHier = 25;
                END;
                nCmiValueMax = nCmiArray(25) ;
                sRugMax = 'ES2';
                nRugMax = 25;
             END;

             *Handle non-therapy classification;
             IF nCmiArray(25) > nCmiValueMax_NT
             THEN DO;
                IF nCmiValueMax_NT = -9999
                THEN DO;
                   nCmiValueHier_NT = nCmiArray(25);
                   sRugHier_NT = 'ES2';
                   nRugHier_NT = 25;
                END;
                nCmiValueMax_NT = nCmiArray(25) ;
                sRugMax_NT = 'ES2';
                nRugMax_NT =25;
             END;

   END;

   ELSE IF     (i_ext_serv_level = 1)
   THEN DO;
             *Handle normal classification;
             IF nCmiArray(26) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(26);
                   sRugHier = 'ES1';
                   nRugHier = 26;
                END;
                nCmiValueMax = nCmiArray(26) ;
                sRugMax = 'ES1';
                nRugMax = 26;
             END;

             *Handle non-therapy classification;
             IF nCmiArray(26) > nCmiValueMax_NT
             THEN DO;
                IF nCmiValueMax_NT = -9999
                THEN DO;
                   nCmiValueHier_NT = nCmiArray(26);
                   sRugHier_NT = 'ES1';
                   nRugHier_NT = 26;
                END;
                nCmiValueMax_NT = nCmiArray(26) ;
                sRugMax_NT = 'ES1';
                nRugMax_NT =26;
             END;

   END;

END;
*End of Classification into Extensive Services Groups for all models;
*===============================;


/*===============================
Begin Classification into 48-group model Rehabilitation groups.
*/
IF sModel = '48' 
THEN DO;

/*
Classify into 48-group model Rehabilitation groups if any rehabilitation
intensity  indicator (ultra high, very high, high, medium or low) 
is present. 

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax.
*/

IF ( (l_rehab_ultra = 1) OR
     (l_rehab_very = 1) OR 
     (l_rehab_high = 1) OR 
     (l_rehab_medium = 1) OR
     (l_rehab_low = 1) )
THEN DO;

   IF     (15 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF nCmiArray(27) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(27);
                   sRugHier = 'RAE';
                   nRugHier = 27;
                END;
                nCmiValueMax = nCmiArray(27) ;
                sRugMax = 'RAE';
                nRugMax = 27;
             END;
   END;

   ELSE IF (11 <= i_adl_tot  AND  i_adl_tot <= 14)
   THEN DO;
             IF nCmiArray(28) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(28);
                   sRugHier = 'RAD';
                   nRugHier = 28;
                END;
                nCmiValueMax = nCmiArray(28) ;
                sRugMax = 'RAD';
                nRugMax = 28;
             END;
   END;

   ELSE IF ( 6 <= i_adl_tot  AND  i_adl_tot <= 10)
   THEN DO;
             IF nCmiArray(29) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(29);
                   sRugHier = 'RAC';
                   nRugHier = 29;
                END;
                nCmiValueMax = nCmiArray(29);
                sRugMax = 'RAC';
                nRugMax =29;
             END;
   END;

   ELSE IF ( 2 <= i_adl_tot  AND  i_adl_tot <= 5)
   THEN DO;
             IF nCmiArray(30) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(30);
                   sRugHier = 'RAB';
                   nRugHier = 30;
                END;
                nCmiValueMax = nCmiArray(30);
                sRugMax = 'RAB';
                nRugMax = 30;
             END;
   END;

   ELSE IF ( 0 <= i_adl_tot  AND  i_adl_tot <= 1)
   THEN DO;
             IF nCmiArray(31) > nCmiValueMax
             THEN DO;
                IF nCmiValueMax = -9999
                THEN DO;
                   nCmiValueHier = nCmiArray(31);
                   sRugHier = 'RAA';
                   nRugHier = 31;
                END;
                nCmiValueMax = nCmiArray(31);
                sRugMax = 'RAA';
                nRugMax = 31;
             END;
   END;
END;

*End Classification into 48-group model Rehabilitation groups.;
END;
*===============================;


/*===============================
Begin Classification into Special Care High for all models (66-,
57-, and 48-groups)

   Classify if Special Care High category indicator (l_spec_high) is
   present.  This indicator is present if both Special Care High clinical
   qualifiers are present and the ADL score is 2 or more.

   Note that residents who have Special Care High clinical qualifiers but
   have too low an ADL score (0 or 1) are classified as Clinically Complex
   rather than Special Care High. 

   Special Care High groups are split out on the basis of ADL score and 
   then depression.

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax.

   Both normal and non-therapy classifications are made for the Special
   Care High groups. 
*/

IF  (l_special_high) 
THEN DO;

   IF     (15 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF l_depression = 1
             THEN DO;
                *Handle depression subgroup;
                *Handle normal classification;
                IF nCmiArray(32) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(32);
                      sRugHier = 'HE2';
                      nRugHier = 32;
                   END;
                   nCmiValueMax = nCmiArray(32);
                   sRugMax = 'HE2';
                   nRugMax = 32;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(32) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(32);
                      sRugHier_NT = 'HE2';
                      nRugHier_NT= 32;
                   END;
                   nCmiValueMax_NT = nCmiArray(32);
                   sRugMax_NT = 'HE2';
                   nRugMax_NT = 32;
                END;

             END;
             ELSE DO; 
                *Handle non-depression subgroup;
                *Handle normal classification;
                IF nCmiArray(33) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(33);
                      sRugHier = 'HE1';
                      nRugHier = 33;
                   END;
                   nCmiValueMax = nCmiArray(33);
                   sRugMax = 'HE1';
                   nRugMax =33;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(33) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(33);
                      sRugHier_NT = 'HE1';
                      nRugHier_NT =33;
                   END;
                   nCmiValueMax_NT = nCmiArray(33);
                   sRugMax_NT = 'HE1';
                   nRugMax_NT = 33;
                END;

             END;
   END;

   ELSE IF     (11 <= i_adl_tot  AND  i_adl_tot <= 14)
   THEN DO;
             IF l_depression = 1
             THEN DO;
                *Handle depression subgroup;
                *Handle normal classification;
                IF nCmiArray(34) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(34);
                      sRugHier = 'HD2';
                      nRugHier =34;
                   END;
                   nCmiValueMax = nCmiArray(34);
                   sRugMax = 'HD2';
                   nRugMax = 34;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(34) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(34);
                      sRugHier_NT = 'HD2';
                      nRugHier_NT= 34;
                   END;
                   nCmiValueMax_NT = nCmiArray(34);
                   sRugMax_NT = 'HD2';
                   nRugMax_NT = 34;
                END;

             END;
             ELSE DO; 
                *Handle non-depression subgroup;
                *Handle normal classification;
                IF nCmiArray(35) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(35);
                      sRugHier = 'HD1';
                      nRugHier = 35;
                   END;
                   nCmiValueMax = nCmiArray(35);
                   sRugMax = 'HD1';
                   nRugMax =35;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(35) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(35);
                      sRugHier_NT = 'HD1';
                      nRugHier_NT =35;
                   END;
                   nCmiValueMax_NT = nCmiArray(35);
                   sRugMax_NT = 'HD1';
                   nRugMax_NT = 35;
                END;

             END;
   END;

   ELSE IF     ( 6 <= i_adl_tot  AND  i_adl_tot <= 10)
   THEN DO;
             IF l_depression = 1
             THEN DO;
                *Handle depression subgroup;
                *Handle normal classification;
                IF nCmiArray(36) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(36);
                      sRugHier = 'HC2';
                      nRugHier = 36;
                   END;
                   nCmiValueMax = nCmiArray(36);
                   sRugMax = 'HC2';
                   nRugMax = 36;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(36) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(36);
                      sRugHier_NT = 'HC2';
                      nRugHier_NT= 36;
                   END;
                   nCmiValueMax_NT = nCmiArray(36);
                   sRugMax_NT = 'HC2';
                   nRugMax_NT = 36;
                END;

             END;
             ELSE DO; 
                *Handle non-depression subgroup;
                *Handle normal classification;
                IF nCmiArray(37) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(37);
                      sRugHier = 'HC1';
                      nRugHier = 37;
                   END;
                   nCmiValueMax = nCmiArray(37);
                   sRugMax = 'HC1';
                   nRugMax = 37;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(37) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(37);
                      sRugHier_NT = 'HC1';
                      nRugHier_NT =37;
                   END;
                   nCmiValueMax_NT = nCmiArray(37);
                   sRugMax_NT = 'HC1';
                   nRugMax_NT = 37;
                END;

             END;
   END;

   ELSE IF     ( 2 <= i_adl_tot  AND  i_adl_tot <=  5)
   THEN DO;
             IF l_depression = 1
             THEN DO;
                *Handle depression subgroup;
                *Handle normal classification;
                IF nCmiArray(38) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(38);
                      sRugHier = 'HB2';
                      nRugHier = 38;
                   END;
                   nCmiValueMax = nCmiArray(38);
                   sRugMax = 'HB2';
                   nRugMax = 38;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(38) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(38);
                      sRugHier_NT = 'HB2';
                      nRugHier_NT= 38;
                   END;
                   nCmiValueMax_NT = nCmiArray(38);
                   sRugMax_NT = 'HB2';
                   nRugMax_NT = 38;
                END;

             END;
             ELSE DO; 
                *Handle non-depression subgroup;
                *Handle normal classification;
                IF nCmiArray(39) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(39);
                      sRugHier = 'HB1';
                      nRugHier = 39;
                   END;
                   nCmiValueMax = nCmiArray(39);
                   sRugMax = 'HB1';
                   nRugMax = 39;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(39) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(39);
                      sRugHier_NT = 'HB1';
                      nRugHier_NT =39;
                   END;
                   nCmiValueMax_NT = nCmiArray(39);
                   sRugMax_NT = 'HB1';
                   nRugMax_NT = 39;
                END;

             END;
   END;

*End Classification into Special Care High groups.;
END;
*===============================;


/*===============================
Begin Classification into Special Care Low for all models (66-,
57-, and 48-groups)

   Classify if Special Care Low category indicator (l_spec_low) is
   present.  This indicator is present if both Special Care Low clinical
   qualifiers are present and the ADL score is 2 or more.

   Note that residents who have Special Care Low clinical qualifiers but
   have too low an ADL score (0 or 1) are classified as Clinically Complex
   rather than Special Care Low. 

   Special Care Low groups are split out on the basis of ADL score and 
   then depression.

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax.

   Both normal and non-therapy classifications are made for the Special
   Care Low groups. 
*/

IF  (l_special_low) 
THEN DO;

   IF     (15 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF l_depression = 1
             THEN DO;
                *Handle depression subgroup;
                *Handle normal classification;
                IF nCmiArray(40) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(40);
                      sRugHier = 'LE2';
                      nRugHier = 40;
                   END;
                   nCmiValueMax = nCmiArray(40);
                   sRugMax = 'LE2';
                   nRugMax = 40;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(40) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(40);
                      sRugHier_NT = 'LE2';
                      nRugHier_NT= 40;
                   END;
                   nCmiValueMax_NT = nCmiArray(40);
                   sRugMax_NT = 'LE2';
                   nRugMax_NT = 40;
                END;

             END;
             ELSE DO; 
                *Handle non-depression subgroup;
                *Handle normal classification;
                IF nCmiArray(41) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(41);
                      sRugHier = 'LE1';
                      nRugHier = 41;
                   END;
                   nCmiValueMax = nCmiArray(41);
                   sRugMax = 'LE1';
                   nRugMax = 41;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(41) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(41);
                      sRugHier_NT = 'LE1';
                      nRugHier_NT = 41;
                   END;
                   nCmiValueMax_NT = nCmiArray(41);
                   sRugMax_NT = 'LE1';
                   nRugMax_NT = 41;
                END;

             END;
   END;

   ELSE IF     (11 <= i_adl_tot  AND  i_adl_tot <= 14)
   THEN DO;
             IF l_depression = 1
             THEN DO;
                *Handle depression subgroup;
                *Handle normal classification;
                IF nCmiArray(42) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(42);
                      sRugHier = 'LD2';
                      nRugHier = 42;
                   END;
                   nCmiValueMax = nCmiArray(42);
                   sRugMax = 'LD2';
                   nRugMax = 42;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(42) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(42);
                      sRugHier_NT = 'LD2';
                      nRugHier_NT= 42;
                   END;
                   nCmiValueMax_NT = nCmiArray(42);
                   sRugMax_NT = 'LD2';
                   nRugMax_NT = 42;
                END;

             END;
             ELSE DO; 
                *Handle non-depression subgroup;
                *Handle normal classification;
                IF nCmiArray(43) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(43);
                      sRugHier = 'LD1';
                      nRugHier = 43;
                   END;
                   nCmiValueMax = nCmiArray(43);
                   sRugMax = 'LD1';
                   nRugMax = 43;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(43) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(43);
                      sRugHier_NT = 'LD1';
                      nRugHier_NT =43;
                   END;
                   nCmiValueMax_NT = nCmiArray(43);
                   sRugMax_NT = 'LD1';
                   nRugMax_NT = 43;
                END;

             END;
   END;

   ELSE IF     ( 6 <= i_adl_tot  AND  i_adl_tot <= 10)
   THEN DO;
             IF l_depression = 1
             THEN DO;
                *Handle depression subgroup;
                *Handle normal classification;
                IF nCmiArray(44) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(44);
                      sRugHier = 'LC2';
                      nRugHier = 44;
                   END;
                   nCmiValueMax = nCmiArray(44);
                   sRugMax = 'LC2';
                   nRugMax = 44;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(44) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(44);
                      sRugHier_NT = 'LC2';
                      nRugHier_NT= 44;
                   END;
                   nCmiValueMax_NT = nCmiArray(44);
                   sRugMax_NT = 'LC2';
                   nRugMax_NT = 44;
                END;

             END;
             ELSE DO; 
                *Handle non-depression subgroup;
                *Handle normal classification;
                IF nCmiArray(45) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(45);
                      sRugHier = 'LC1';
                      nRugHier = 45;
                   END;
                   nCmiValueMax = nCmiArray(45);
                   sRugMax = 'LC1';
                   nRugMax = 45;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(45) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(45);
                      sRugHier_NT = 'LC1';
                      nRugHier_NT =45;
                   END;
                   nCmiValueMax_NT = nCmiArray(45);
                   sRugMax_NT = 'LC1';
                   nRugMax_NT = 45;
                END;

             END;
   END;

   ELSE IF     ( 2 <= i_adl_tot  AND  i_adl_tot <=  5)
   THEN DO;
             IF l_depression = 1
             THEN DO;
                *Handle depression subgroup;
                *Handle normal classification;
                IF nCmiArray(46) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(46);
                      sRugHier = 'LB2';
                      nRugHier = 46;
                   END;
                   nCmiValueMax = nCmiArray(46);
                   sRugMax = 'LB2';
                   nRugMax = 46;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(46) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(46);
                      sRugHier_NT = 'LB2';
                      nRugHier_NT= 46;
                   END;
                   nCmiValueMax_NT = nCmiArray(46);
                   sRugMax_NT = 'LB2';
                   nRugMax_NT = 46;
                END;

             END;
             ELSE DO; 
                *Handle non-depression subgroup;
                *Handle normal classification;
                IF nCmiArray(47) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(47);
                      sRugHier = 'LB1';
                      nRugHier = 47;
                   END;
                   nCmiValueMax = nCmiArray(47);
                   sRugMax = 'LB1';
                   nRugMax = 47;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(47) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(47);
                      sRugHier_NT = 'LB1';
                      nRugHier_NT = 47;
                   END;
                   nCmiValueMax_NT = nCmiArray(47);
                   sRugMax_NT = 'LB1';
                   nRugMax_NT = 47;
                END;

             END;
   END;

*End Classification into Special Care Low groups.;
END;
*===============================;


/*===============================
Begin Classification into Clinically Complex for all models (66-,
57-, and 48-groups)

   Classify if Clinically Complex category indicator (l_clin_complex) is
   present.  Also classify if Extensive Services are received
   (i_ext_serv_level >= 1) and the ADL score is 0 or 1 (i_adl_tot <= 1).
   Also classify if a Special Care High condition is present 
   (l_sp_high_cond = 1) and the ADL score is 0 or 1 (i_adl_tot <= 1).
   Also classify if a Special Care Low condition is present 
   (l_sp_low_cond = 1) and the ADL score is 0 or 1 (i_adl_tot <= 1).

   Clinically Complex groups are split out on the basis of ADL score and 
   then depression.

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax.

   Both normal and non-therapy classifications are made for the Clinically
   Complex groups. 
*/

IF ( (l_clin_complex = 1) OR
     ( (i_ext_serv_level >= 1) AND (i_adl_tot <= 1) ) OR 
     ( (l_sp_high_cond = 1) AND (i_adl_tot <= 1) ) OR 
     ( (l_sp_low_cond = 1) AND (i_adl_tot <= 1) ) ) 
THEN DO;

   IF     (15 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF l_depression = 1
             THEN DO;
                *Handle depression subgroup;
                *Handle normal classification;
                IF nCmiArray(48) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(48);
                      sRugHier = 'CE2';
                      nRugHier = 48;
                   END;
                   nCmiValueMax = nCmiArray(48);
                   sRugMax = 'CE2';
                   nRugMax = 48;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(48) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(48);
                      sRugHier_NT = 'CE2';
                      nRugHier_NT= 48;
                   END;
                   nCmiValueMax_NT = nCmiArray(48);
                   sRugMax_NT = 'CE2';
                   nRugMax_NT = 48;
                END;

             END;
             ELSE DO; 
                *Handle non-depression subgroup;
                *Handle normal classification;
                IF nCmiArray(49) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(49);
                      sRugHier = 'CE1';
                      nRugHier = 49;
                   END;
                   nCmiValueMax = nCmiArray(49);
                   sRugMax = 'CE1';
                   nRugMax = 49;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(49) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(49);
                      sRugHier_NT = 'CE1';
                      nRugHier_NT =49;
                   END;
                   nCmiValueMax_NT = nCmiArray(49);
                   sRugMax_NT = 'CE1';
                   nRugMax_NT = 49;
                END;

             END;
   END;

   ELSE IF     (11 <= i_adl_tot  AND  i_adl_tot <= 14)
   THEN DO;
             IF l_depression = 1
             THEN DO;
                *Handle depression subgroup;
                *Handle normal classification;
                IF nCmiArray(50) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(50);
                      sRugHier = 'CD2';
                      nRugHier = 50;
                   END;
                   nCmiValueMax = nCmiArray(50);
                   sRugMax = 'CD2';
                   nRugMax = 50;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(50) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(50);
                      sRugHier_NT = 'CD2';
                      nRugHier_NT= 50;
                   END;
                   nCmiValueMax_NT = nCmiArray(50);
                   sRugMax_NT = 'CD2';
                   nRugMax_NT = 50;
                END;

             END;
             ELSE DO; 
                *Handle non-depression subgroup;
                *Handle normal classification;
                IF nCmiArray(51) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(51);
                      sRugHier = 'CD1';
                      nRugHier = 51;
                   END;
                   nCmiValueMax = nCmiArray(51);
                   sRugMax = 'CD1';
                   nRugMax = 51;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(51) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(51);
                      sRugHier_NT = 'CD1';
                      nRugHier_NT =51;
                   END;
                   nCmiValueMax_NT = nCmiArray(51);
                   sRugMax_NT = 'CD1';
                   nRugMax_NT = 51;
                END;

             END;
   END;

   ELSE IF     ( 6 <= i_adl_tot  AND  i_adl_tot <= 10)
   THEN DO;
             IF l_depression = 1
             THEN DO;
                *Handle depression subgroup;
                *Handle normal classification;
                IF nCmiArray(52) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(52);
                      sRugHier = 'CC2';
                      nRugHier = 52;
                   END;
                   nCmiValueMax = nCmiArray(52);
                   sRugMax = 'CC2';
                   nRugMax = 52;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(52) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(52);
                      sRugHier_NT = 'CC2';
                      nRugHier_NT= 52;
                   END;
                   nCmiValueMax_NT = nCmiArray(52);
                   sRugMax_NT = 'CC2';
                   nRugMax_NT = 52;
                END;

             END;
             ELSE DO; 
                *Handle non-depression subgroup;
                *Handle normal classification;
                IF nCmiArray(53) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(53);
                      sRugHier = 'CC1';
                      nRugHier = 53;
                   END;
                   nCmiValueMax = nCmiArray(53);
                   sRugMax = 'CC1';
                   nRugMax = 53;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(53) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(53);
                      sRugHier_NT = 'CC1';
                      nRugHier_NT =53;
                   END;
                   nCmiValueMax_NT = nCmiArray(53);
                   sRugMax_NT = 'CC1';
                   nRugMax_NT = 53;
                END;

             END;
   END;

   ELSE IF     ( 2 <= i_adl_tot  AND  i_adl_tot <=  5)
   THEN DO;
             IF l_depression = 1
             THEN DO;
                *Handle depression subgroup;
                *Handle normal classification;
                IF nCmiArray(54) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(54);
                      sRugHier = 'CB2';
                      nRugHier = 54;
                   END;
                   nCmiValueMax = nCmiArray(54);
                   sRugMax = 'CB2';
                   nRugMax = 54;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(54) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(54);
                      sRugHier_NT = 'CB2';
                      nRugHier_NT= 54;
                   END;
                   nCmiValueMax_NT = nCmiArray(54);
                   sRugMax_NT = 'CB2';
                   nRugMax_NT = 54;
                END;

             END;
             ELSE DO; 
                *Handle non-depression subgroup;
                *Handle normal classification;
                IF nCmiArray(55) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(55);
                      sRugHier = 'CB1';
                      nRugHier = 55;
                   END;
                   nCmiValueMax = nCmiArray(55);
                   sRugMax = 'CB1';
                   nRugMax = 55;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(55) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(55);
                      sRugHier_NT = 'CB1';
                      nRugHier_NT =55;
                   END;
                   nCmiValueMax_NT = nCmiArray(55);
                   sRugMax_NT = 'CB1';
                   nRugMax_NT = 55;
                END;

             END;
   END;

   ELSE IF     ( 0 <= i_adl_tot  AND  i_adl_tot <=  1) 
   THEN DO;
             IF l_depression = 1
             THEN DO;
                *Handle depression subgroup;
                *Handle normal classification;
                IF nCmiArray(56) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(56);
                      sRugHier = 'CA2';
                      nRugHier = 56;
                   END;
                   nCmiValueMax = nCmiArray(56);
                   sRugMax = 'CA2';
                   nRugMax = 56;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(56) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(56);
                      sRugHier_NT = 'CA2';
                      nRugHier_NT= 56;
                   END;
                   nCmiValueMax_NT = nCmiArray(56);
                   sRugMax_NT = 'CA2';
                   nRugMax_NT = 56;
                END;

             END;
             ELSE DO; 
                *Handle non-depression subgroup;
                *Handle normal classification;
                IF nCmiArray(57) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(57);
                      sRugHier = 'CA1';
                      nRugHier = 57;
                   END;
                   nCmiValueMax = nCmiArray(57);
                   sRugMax = 'CA1';
                   nRugMax = 57;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(57) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(57);
                      sRugHier_NT = 'CA1';
                      nRugHier_NT = 57;
                   END;
                   nCmiValueMax_NT = nCmiArray(57);
                   sRugMax_NT = 'CA1';
                   nRugMax_NT = 57;
                END;

             END;
   END;

*End Classification into Clinically Complex groups.;
END;
*===============================;

/*===============================
Begin Classification into Behavior Symptoms and Cognitive Performance for
all models (66-, 57-, and 48-groups)

   Classify if Behavior Symptoms and Cognitive Performance category
   indicator (l_beh_cog) is present.  This indicator is present if the ADL
   score with either behavior symptoms or cognitive impairment.

   Behavior Symptoms and Cognitive Performance groups are split out on the
   basis of ADL score and then restorative nursing.

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax.

   Both normal and non-therapy classifications are made for the Special
   Care Low groups. 
*/

IF ( l_beh_cog = 1) 
THEN DO;

   IF          ( 2 <= i_adl_tot  AND  i_adl_tot <=  5)
   THEN DO;
             IF l_rnursing = 1
             THEN DO;
                *Handle restorative nursing subgroup;
                *Handle normal classification;
                IF nCmiArray(58) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(58);
                      sRugHier = 'BB2';
                      nRugHier = 58;
                   END;
                   nCmiValueMax = nCmiArray(58);
                   sRugMax = 'BB2';
                   nRugMax = 58;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(58) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(58);
                      sRugHier_NT = 'BB2';
                      nRugHier_NT= 58;
                   END;
                   nCmiValueMax_NT = nCmiArray(58);
                   sRugMax_NT = 'BB2';
                   nRugMax_NT = 58;
                END;

             END;
             ELSE DO; 
                *Handle no restorative nursing subgroup;
                *Handle normal classification;
                IF nCmiArray(59) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(59);
                      sRugHier = 'BB1';
                      nRugHier = 59;
                   END;
                   nCmiValueMax = nCmiArray(59);
                   sRugMax = 'BB1';
                   nRugMax = 59;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(59) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(59);
                      sRugHier_NT = 'BB1';
                      nRugHier_NT =59;
                   END;
                   nCmiValueMax_NT = nCmiArray(59);
                   sRugMax_NT = 'BB1';
                   nRugMax_NT = 59;
                END;

             END;
   END;

   ELSE IF     ( 0 <= i_adl_tot  AND  i_adl_tot <=  1) 
   THEN DO;
             IF l_rnursing = 1
             THEN DO;
                *Handle restorative nursing subgroup;
                *Handle normal classification;
                IF nCmiArray(60) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(60);
                      sRugHier = 'BA2';
                      nRugHier = 60;
                   END;
                   nCmiValueMax = nCmiArray(60);
                   sRugMax = 'BA2';
                   nRugMax = 60;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(60) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(60);
                      sRugHier_NT = 'BA2';
                      nRugHier_NT= 60;
                   END;
                   nCmiValueMax_NT = nCmiArray(60);
                   sRugMax_NT = 'BA2';
                   nRugMax_NT = 60;
                END;

             END;
             ELSE DO; 
                *Handle no restorative nursing subgroup;
                *Handle normal classification;
                IF nCmiArray(61) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(61);
                      sRugHier = 'BA1';
                      nRugHier = 61;
                   END;
                   nCmiValueMax = nCmiArray(61);
                   sRugMax = 'BA1';
                   nRugMax = 61;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(61) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(61);
                      sRugHier_NT = 'BA1';
                      nRugHier_NT = 61;
                   END;
                   nCmiValueMax_NT = nCmiArray(61);
                   sRugMax_NT = 'BA1';
                   nRugMax_NT = 61;
                END;

             END;
   END;

*End Classification into Behavior Symptoms and Cognitive Performance
   groups.;
END;
*===============================;


/*===============================
Begin Classification into Reduced Physical Function groups for all models
(66-, 57-, and 48-groups)

   All residents qualify for the Reduced Physical Function category.

   Reduced Physical Function groups are split out on the basis of ADL score
   and then restorative nursing.

   Assign a hierarchical classification if a previous hierarchical
   classification has not been made (indicated by nCmiValueMax being at
   the default negative value).

   Assign an index maximizing classification if the CMI value in the CMI
   array (nCmiArray) for the current group is greater than the previously
   assigned value in nCmiValueMax.

   Both normal and non-therapy classifications are made for the Special
   Care Low groups. 
*/

   IF     (15 <= i_adl_tot  AND  i_adl_tot <= 16)
   THEN DO;
             IF l_rnursing = 1
             THEN DO;
                *Handle restorative nursing subgroup;
                *Handle normal classification;
                IF nCmiArray(62) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(62);
                      sRugHier = 'PE2';
                      nRugHier = 62;
                   END;
                   nCmiValueMax = nCmiArray(62);
                   sRugMax = 'PE2';
                   nRugMax = 62;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(62) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(62);
                      sRugHier_NT = 'PE2';
                      nRugHier_NT = 62;
                   END;
                   nCmiValueMax_NT = nCmiArray(62);
                   sRugMax_NT = 'PE2';
                   nRugMax_NT = 62;
                END;

             END;
             ELSE DO; 
                *Handle no restorative nursing subgroup;
                *Handle normal classification;
                IF nCmiArray(63) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(63);
                      sRugHier = 'PE1';
                      nRugHier = 63;
                   END;
                   nCmiValueMax = nCmiArray(63);
                   sRugMax = 'PE1';
                   nRugMax = 63;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(63) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(63);
                      sRugHier_NT = 'PE1';
                      nRugHier_NT = 63;
                   END;
                   nCmiValueMax_NT = nCmiArray(63);
                   sRugMax_NT = 'PE1';
                   nRugMax_NT = 63;
                END;

             END;
   END;

   ELSE IF     (11 <= i_adl_tot  AND  i_adl_tot <= 14)
   THEN DO;
             IF l_rnursing = 1
             THEN DO;
                *Handle restorative nursing subgroup;
                *Handle normal classification;
                IF nCmiArray(64) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(64);
                      sRugHier = 'PD2';
                      nRugHier = 64;
                   END;
                   nCmiValueMax = nCmiArray(64);
                   sRugMax = 'PD2';
                   nRugMax = 64;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(64) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(64);
                      sRugHier_NT = 'PD2';
                      nRugHier_NT= 64;
                   END;
                   nCmiValueMax_NT = nCmiArray(64);
                   sRugMax_NT = 'PD2';
                   nRugMax_NT = 64;
                END;

             END;
             ELSE DO; 
                *Handle no restorative nursing subgroup;
                *Handle normal classification;
                IF nCmiArray(65) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(65);
                      sRugHier = 'PD1';
                      nRugHier = 65;
                   END;
                   nCmiValueMax = nCmiArray(65);
                   sRugMax = 'PD1';
                   nRugMax = 65;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(65) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(65);
                      sRugHier_NT = 'PD1';
                      nRugHier_NT = 65;
                   END;
                   nCmiValueMax_NT = nCmiArray(65);
                   sRugMax_NT = 'PD1';
                   nRugMax_NT = 65;
                END;

             END;
   END;

   ELSE IF     ( 6 <= i_adl_tot  AND  i_adl_tot <= 10)
   THEN DO;
             IF l_rnursing = 1
             THEN DO;
                *Handle restorative nursing subgroup;
                *Handle normal classification;
                IF nCmiArray(66) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(66);
                      sRugHier = 'PC2';
                      nRugHier = 66;
                   END;
                   nCmiValueMax = nCmiArray(66);
                   sRugMax = 'PC2';
                   nRugMax = 66;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(66) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(66);
                      sRugHier_NT = 'PC2';
                      nRugHier_NT = 66;
                   END;
                   nCmiValueMax_NT = nCmiArray(66);
                   sRugMax_NT = 'PC2';
                   nRugMax_NT = 66;
                END;

             END;
             ELSE DO; 
                *Handle no restorative nursing subgroup;
                *Handle normal classification;
                IF nCmiArray(67) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(67);
                      sRugHier = 'PC1';
                      nRugHier = 67;
                   END;
                   nCmiValueMax = nCmiArray(67);
                   sRugMax = 'PC1';
                   nRugMax = 67;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(67) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(67);
                      sRugHier_NT = 'PC1';
                      nRugHier_NT = 67;
                   END;
                   nCmiValueMax_NT = nCmiArray(67);
                   sRugMax_NT = 'PC1';
                   nRugMax_NT = 67;
                END;

             END;
   END;

   ELSE IF     ( 2 <= i_adl_tot  AND  i_adl_tot <=  5)
   THEN DO;
             IF l_rnursing = 1
             THEN DO;
                *Handle restorative nursing subgroup;
                *Handle normal classification;
                IF nCmiArray(68) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(68);
                      sRugHier = 'PB2';
                      nRugHier = 68;
                   END;
                   nCmiValueMax = nCmiArray(68);
                   sRugMax = 'PB2';
                   nRugMax = 68;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(68) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(68);
                      sRugHier_NT = 'PB2';
                      nRugHier_NT = 68;
                   END;
                   nCmiValueMax_NT = nCmiArray(68);
                   sRugMax_NT = 'PB2';
                   nRugMax_NT = 68;
                END;

             END;
             ELSE DO; 
                *Handle no restorative nursing subgroup;
                *Handle normal classification;
                IF nCmiArray(69) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(69);
                      sRugHier = 'PB1';
                      nRugHier = 69;
                   END;
                   nCmiValueMax = nCmiArray(69);
                   sRugMax = 'PB1';
                   nRugMax = 69;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(69) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(69);
                      sRugHier_NT = 'PB1';
                      nRugHier_NT = 69;
                   END;
                   nCmiValueMax_NT = nCmiArray(69);
                   sRugMax_NT = 'PB1';
                   nRugMax_NT = 69;
                END;

             END;
   END;

   ELSE IF     ( 0 <= i_adl_tot  AND  i_adl_tot <=  1) 
   THEN DO;
             IF l_rnursing = 1
             THEN DO;
                *Handle restorative nursing subgroup;
                *Handle normal classification;
                IF nCmiArray(70) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(70);
                      sRugHier = 'PA2';
                      nRugHier = 70;
                   END;
                   nCmiValueMax = nCmiArray(70);
                   sRugMax = 'PA2';
                   nRugMax = 70;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(70) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(70);
                      sRugHier_NT = 'PA2';
                      nRugHier_NT= 70;
                   END;
                   nCmiValueMax_NT = nCmiArray(70);
                   sRugMax_NT = 'PA2';
                   nRugMax_NT = 70;
                END;

             END;
             ELSE DO; 
                *Handle no restorative nursing subgroup;
                *Handle normal classification;
                IF nCmiArray(71) > nCmiValueMax
                THEN DO;
                   IF nCmiValueMax = -9999
                   THEN DO;
                      nCmiValueHier = nCmiArray(71);
                      sRugHier = 'PA1';
                      nRugHier = 71;
                   END;
                   nCmiValueMax = nCmiArray(71);
                   sRugMax = 'PA1';
                   nRugMax = 71;
                END;

                *Handle non-therapy classification;      
                IF nCmiArray(71) > nCmiValueMax_NT
                THEN DO;
                   IF nCmiValueMax_NT = -9999
                   THEN DO;
                      nCmiValueHier_NT = nCmiArray(71);
                      sRugHier_NT = 'PA1';
                      nRugHier_NT = 71;
                   END;
                   nCmiValueMax_NT = nCmiArray(71);
                   sRugMax_NT = 'PA1';
                   nRugMax_NT = 71;
                END;

             END;
   END;

*End Classification into Reduced Physical Function groups.;
*===============================;

/*
Adjustment in RUG group for a start of therapy OMRA (A0310C = 1 or 3).
*/
IF ( (A0310C = '1') OR (A0310C = '3') )
   THEN DO;
      *V1.03.0 CHANGE to include new sRehabType = 'MCAR3';
      *Handle Medicare classification (sRehabType = 'MCARE', 'MCAR2', 
         or MCAR3);
      IF ((sRehabType = 'MCARE') OR (sRehabType = 'MCAR2') OR 
       (sRehabType = 'MCAR3'))
         THEN DO;
            /*
            If the normal index maximizing RUG group is not a 
            Rehabilitation Plus Extensive Services or a Rehabilitation 
            Group (first letter of the group is not 'R'), then reset 
            all normal and non-therapy RUG results to the AAA default 
            group.  Also set iError = 5 to reject the record.
            */   
            IF (SUBSTR(sRugMax, 1, 1) NOT = 'R')
               THEN DO;
                  sRugHier = 'AAA';
                  nRugHier = 72;
                  nCmiValueHier = nCmiArray(72);
                  sRugMax = 'AAA';
                  nRugMax = 72;
                  nCmiValueMax = nCmiArray(72);
                  sRugHier_NT = 'AAA';
                  nRugHier_NT = 72;
                  nCmiValueHier_NT = nCmiArray(72);
                  sRugMax_NT = 'AAA';
                  nRugMax_NT = 72;
                  nCmiValueMax_NT = nCmiArray(72);

                  iError = 5;
             END;
           
            /*
            If a start of therapy OMRA gives a Medicare index maximized 
            Rehabilitation Plus Extensive Services or a Rehabilitation
            Group and is not combined with an OBRA assessment or other
            PPS assessment (including an end of therapy OMRA) then reset 
            all non-therapy RUG results to the AAA default group.
            */
            ELSE IF ( (A0310A = '99') AND 
               ( (A0310B = '99') OR (A0310B = '07') ) AND
               (A0310C = '1') AND 
               (A0310D NOT = '1') ) 
                  THEN DO;    
                     sRugHier_NT = 'AAA';
                     nRugHier_NT = 72;
                     nCmiValueHier_NT = nCmiArray(72);
                     sRugMax_NT = 'AAA';
                     nRugMax_NT = 72;
                     nCmiValueMax_NT = nCmiArray(72);
            END;

      END;


      /*
      Handle non-Medicare classification (sRehabType will always = 
      'OTHER') here;
      */
      ELSE
         DO;
            /*
            The non-Medicare classification is not required to give an  
            index maximized Rehabilitation Plus Extensive Services or 
            a Rehabilitation Group on a start of therapy OMRA.  The only
            restriction for non-Medicare classification is that the 
            the classification cannot be determined on a standalone 
            start of therapy OMRA that is not combined with an OBRA 
            assessment or other PPS assessment (including an end of 
            therapy OMRA).  A standalone start of therapy OMRA does
            not include all of the MDS items required for 
            classification.  For a standalone start of therapy OMRA, 
            reset all normal and non-therapy RUG results to the AAA 
            default group. Do not set iError = 5.
            */  
            IF ( (A0310A = '99') AND 
               ( (A0310B = '99') OR (A0310B = '07') ) AND
               (A0310C = '1') AND 
               (A0310D NOT = '1') ) 
                  THEN DO;    
                     sRugHier = 'AAA';
                     nRugHier = 72;
                     nCmiValueHier = nCmiArray(72);
                     sRugMax = 'AAA';
                     nRugMax = 72;
                     nCmiValueMax = nCmiArray(72);
                     sRugHier_NT = 'AAA';
                     nRugHier_NT = 72;
                     nCmiValueHier_NT = nCmiArray(72);
                     sRugMax_NT = 'AAA';
                     nRugMax_NT = 72;
                     nCmiValueMax_NT = nCmiArray(72);
            END;
      END;
         
END;

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
CALCULATION OF HIPPS ASSESSMENT INDICATOR(AI) code.                                           
     The last two positions of the Medicare HIPPS code (MDS 3.0 items Z0100A
     and Z0150A) represent the Assessment Indicator (AI), identifying the
     assessment type. The AI coding system indicates the different types of
     assessments that define different Medicare Part A PPS payment periods. 

     There are scheduled PPS assessments performed around Day 5, Day 14, 
     Day 30, Day 60, and Day 90 of a Medicare Part A stay.  These scheduled
     assessments establish per diem payment rates for associated standard
     payment periods.  Unscheduled off-cycle assessments are performed when
     the residents condition changes or the level of rehabilitation services
     changes during a standard payment period.  These unscheduled assessments
     can impact the per diem payment rates for days within a standard payment
     period.

     First AI Digit. The first digit identifies scheduled Medicare PPS 
     assessments that establish RUG payment for the standard PPS scheduled
     payment periods. These assessments are PPS 5-day, 14-day, 30-day,
     60-day, 90-day, and readmission/return.  In certain cases, an OBRA
     assessment may also be used to establish payment for the first Medicare
     PPS standard payment period, and the first digit of the AI code will 
     indicate an OBRA assessment used for PPS. 

     Second AI Digit. The second digit identifies unscheduled assessments.
     Unscheduled PPS assessments are conducted in addition to the required
     standard scheduled PPS assessments and include the following OBRA
     unscheduled assessments: Significant Change in Status Assessment (SCSA)
     and Significant Correction to Prior Full Assessment (SCPA), as well as
     the following PPS unscheduled assessments: Start of Therapy Other
     Medicare-required Assessment (OMRA) and End of Therapy OMRA and the
     Swing Bed Clinical Change Assessment (CCA). Unscheduled assessments may
     be required at any time during the residents Part A stay. They may be
     completed as a separate assessment or combined with other assessments
     and, in some instances, will replace the scheduled PPS assessment.

     When an unscheduled assessment replaces a scheduled PPS assessment, the
     assessment establishes the payment rate for the standard payment period
     normally associated with the scheduled PPS assessment.  The assessment
     should indicate both the scheduled PPS assessment being replaced and the
     type of the unscheduled assessment.  For example, if an SCSA replaces the
     PPS 30-day assessment, then MDS item A0310A should be coded as 04
     indicating the SCSA and A0310B coded as 03 indicating that the PPS 30-day
     assessment is being replaced.  Another example would be a Start of
     Therapy OMRA replacing the PPS 14-day assessment.  In this case, A0310C
     should be coded as 1 indicating the Start of Therapy OMRA and A0310B
     coded as 02 indicating that the PPS 14-day assessment is being replaced.

     Whether an unscheduled assessment is a separate assessment, is combined
     with another assessment, or is replacing a scheduled PPS assessment, the
     unscheduled assessment can impact the payment for days within a standard
     payment period.  
*/

/*---------------------------------------------------------------------------
Calculate 1st AI digit (sAI_1) based on item A0310B.
*/

*If unscheduled PPS assessment, then 1st digit = 0;
IF A0310B = '07' THEN sAI_1 = '0';
*If PPS 5-day or readmission/return assessment, then 1st digit = 1;
ELSE IF (A0310B = '01' OR A0310B = '06') THEN sAI_1 = '1';
*IF PPS 14-day assessment, then 1st digit = 2;
ELSE IF A0310B = '02' THEN sAI_1 = '2';
*If PPS 30-day assessment, then 1st digit = 3;
ELSE IF A0310B = '03' THEN sAI_1 = '3';
*If PPS 60-day assessment, then 1st digit = 4;
ELSE IF A0310B = '04' THEN sAI_1 = '4';
*If PPS 90-day assessment, then 1st digit = 5;
ELSE IF A0310B = '05' THEN sAI_1 = '5';
*If OBRA assessment used for PPS, then 1st digit = 6;
ELSE IF (A0310A >= '01' AND A0310A <= '06' AND A0310B = '99')
     THEN sAI_1 = '6';
/*If MDS record type does not support RUG-IV determination (e.g.,
     record is entry or discharge), then 1st digit = X
*/ 
ELSE sAI_1 ='X';

/*---------------------------------------------------------------------------
Calculate 2nd AI digit (sAI_2) based on items A0310A, A0310B, A0310C, and
     A0310D.
*/

sAI_2 = '0';

/*If (1) not PPS assessment or (2) scheduled PPS assessment not combined with
    unscheduled OBRA (significant change or significant correction of
    prior comprehensive) and not combined with swing bed clinical change, and
    not combined with any OMRA, then set 2nd digit to 0.
*/
IF (A0310B = '99') OR
   (  (A0310B >= '01' AND A0310B <= '06') AND 
      (A0310A NOT = '04' AND A0310A NOT = '05' AND A0310D NOT = '1' AND
       A0310C = '0')  )
        THEN sAI_2 = '0';

/*If either unscheduled OBRA (significant change or significant correction 
     of prior comprehensive) or swing bed clinical change, then set
     2nd digit to 1.  Exclude if combined with any OMRA assessment.
*/
ELSE IF (A0310A = '04' OR A0310A = '05' OR A0310D = '1') AND
   (A0310C = '0')  
        THEN sAI_2 = '1';

/*If start of therapy OMRA , then set 2nd digit to 2.  Exclude if combined with
     unscheduled OBRA (significant change or significant correction of prior 
     comprehensive) or combined with swing bed clinical change or combined with
     end of therapy OMRA.  Also exclude if a Medicare short stay assessment.
*/
ELSE IF (A0310C = '1') AND
   (A0310A NOT = '04' AND A0310A NOT = '05' AND A0310D NOT = '1' AND 
      l_Mcare_short_stay = 0)
          THEN sAI_2 = '2';
 
/*If start of therapy OMRA combined with either unscheduled OBRA (significant
     change or significant correction of prior comprehensive) or swing bed
     clinical change, then set 2nd digit to 3.  Exclude if combined with an
     end of therapy OMRA.  Also exclude if Medicare short stay assessment.
*/
ELSE IF (A0310C = '1') AND
   (A0310A = '04' OR A0310A = '05' OR A0310D = '1') AND
   (l_Mcare_short_stay = 0)
        THEN sAI_2 = '3';

/*
If end of therapy OMRA not reporting resumption of therapy (O0450A not = 1),
     then set 2nd digit to 4.  Exclude if combined with a start of therapy
     OMRA.   
*/
ELSE IF (A0310C = '2' AND O0450A NOT = '1')
        THEN sAI_2 = '4';

/*If start of therapy OMRA combined with end of therapy OMRA not reporting
     resumption of therapy (O0450A not = 1), then set 2nd digit to 5.
     Exclude if combined with unscheduled OBRA (significant change or 
     significant correction of prior comprehensive) or combined with swing
     bed clinical change.  Also exclude if a Medicare short stay assessment.
 */
ELSE IF (A0310C = '3' AND O0450A NOT = '1') AND
   (A0310A NOT = '04' AND A0310A NOT = '05' AND A0310D NOT = '1' AND 
      l_Mcare_short_stay = 0)
           THEN sAI_2 = '5';

/*If start of therapy OMRA combined with end of therapy OMRA not reporting
     resumption of therapy (O0450A not = 1), also combined with either
     unscheduled OBRA (significant change or significant correction of
     prior comprehensive) or swing bed clinical change then set 2nd digit
     to 6.  Exclude if a Medicare short stay assessment.
 */
ELSE IF (A0310C = '3' AND O0450A NOT = '1') AND
   (A0310A = '04' OR A0310A = '05' OR A0310D = '1') AND 
   (l_Mcare_short_stay = 0)
           THEN sAI_2 = '6';

/* If end of therapy OMRA reporting resumption of therapy (O0450A = 1),
     then set 2nd digit to 'A'.  Exclude if combined with a start of therapy
     OMRA.   
*/
ELSE IF (A0310C = '2' AND O0450A = '1')
        THEN sAI_2 = 'A';

/* If start of therapy OMRA combined with end of therapy OMRA reporting
     resumption of therapy (O0450A = 1), then set 2nd digit to 'B'.
     Exclude if combined with unscheduled OBRA (significant change or 
     significant correction of prior comprehensive) or combined with swing
     bed clinical change.  Also exclude if a Medicare short stay assessment.
 */
ELSE IF (A0310C = '3' AND O0450A = '1') AND
   (A0310A NOT = '04' AND A0310A NOT = '05' AND A0310D NOT = '1' AND 
      l_Mcare_short_stay = 0)
           THEN sAI_2 = 'B';

/* If start of therapy OMRA combined with end of therapy OMRA reporting
     resumption of therapy (O0450A = 1), also combined with either
     unscheduled OBRA (significant change or significant correction of
     prior comprehensive) or swing bed clinical change then set 2nd digit
     to 'C'.  Exclude if a Medicare short stay assessment.
 */
ELSE IF (A0310C = '3' AND O0450A = '1') AND
   (A0310A = '04' OR A0310A = '05' OR A0310D = '1') AND 
   (l_Mcare_short_stay = 0)
           THEN sAI_2 = 'C';

/* If Change of therapy OMRA, then set 2nd digit to D, even if set
     before.
*/
IF (A0310C = 4)
           THEN sAI_2 = 'D';

/*If Medicare short stay assessment, then set 2nd digit to 7, even if set
     before.
*/
IF (l_Mcare_short_stay = 1)
           THEN sAI_2 = '7';

*Set 2nd digit to blank if first digit is X;
IF sAI_1 = 'X' THEN sAI_2 = ' ';

*Concatenante sAI_1 and sAI_2 into final sAI_code;
sAI_code = sAI_1 || sAI_2;

*Reset AI code to 00 if RUG-IV group is AAA (default group);
IF sRugHier = 'AAA' THEN sAI_code = '00';


SKIPEND:
